package com.gingersoft.gsa.cloud.pay;

import android.os.Parcel;
import android.os.Parcelable;

import com.gingersoft.gsa.cloud.pay.bean.PayMethod;
import com.gingersoft.gsa.cloud.pay.contract.PayMethodContract;

import java.util.List;

import lombok.Data;

/**
 * @作者: bin
 * @創建時間: 2021-04-15 16:46
 * @更新時間: 2021-04-15 16:46
 * @描述:
 */
@Data
public class RequestParms implements Parcelable {

    private PayAction action;
    private final long orderId;
    private final int resturantId;
    private int tableId;
    private String tableExtId;
    private final List<PayMethod> payMethods;
    private final int payType;
    private final double payMoney;
    /**
     * 自定義ID,與POS方綁定交易ID綁定，可用此ID查詢對應的交易記錄
     */
    private String txnId;
    /**
     * 退款相关
     */
    private final String refundReason;
    /**
     * 操作行为回调如支付，退款等
     */
    private CommandCallback commandCallback;
    /**
     * 结果回调
     */
    private ResultCallback resultCallback;

    private RequestParms(Builder builder) {
        this.action = builder.action;
        this.resturantId = builder.resturantId;
        this.orderId = builder.orderId;
        this.tableId = builder.tableId;
        this.tableExtId = builder.tableExtId;
        this.payMethods = builder.payMethods;
        this.payType = builder.payType;
        this.payMoney = builder.payMoney;
        this.refundReason = builder.refundReason;
        this.txnId = builder.txnId;
        this.commandCallback = builder.commandCallback;
        this.resultCallback = builder.resultCallback;
    }

    public PayMethod getCurrPosPaymethod() {
        if (payMethods != null) {
            List<Integer> posPaymethods = PayMethodContract.getPosPaymethods();
            for (PayMethod payMethod : payMethods) {
                for (Integer posPaymethod : posPaymethods) {
                    if (payMethod.getPayType() == posPaymethod) {
                        return payMethod;
                    }
                }
            }
        }
        return null;
    }


    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {

        private PayAction action;
        private long orderId;
        private int resturantId;
        private int tableId;
        private String tableExtId;
        private List<PayMethod> payMethods;
        private int payType;
        private double payMoney;
        private String txnId;
        private String refundReason;
        private CommandCallback commandCallback;
        private ResultCallback resultCallback;

        public Builder action(PayAction action) {
            this.action = action;
            return this;
        }

        public Builder resturantId(int resturantId) {
            this.resturantId = resturantId;
            return this;
        }

        public Builder tableId(int tableId) {
            this.tableId = tableId;
            return this;
        }

        public Builder tableExtId(String tableExtId) {
            this.tableExtId = tableExtId;
            return this;
        }

        public Builder payMethods(List<PayMethod> payMethods) {
            this.payMethods = payMethods;
            return this;
        }

        public Builder payType(int payType) {
            this.payType = payType;
            return this;
        }

        public Builder orderId(long orderId) {
            this.orderId = orderId;
            return this;
        }

        public Builder payMoney(double payMoney) {
            this.payMoney = payMoney;
            return this;
        }

        public Builder txnId(String txnId) {
            this.txnId = txnId;
            return this;
        }

        public Builder refundReason(String refundReason) {
            this.refundReason = refundReason;
            return this;
        }

        public Builder commandCallback(CommandCallback commandCallback) {
            this.commandCallback = commandCallback;
            return this;
        }

        public Builder resultCallback(ResultCallback resultCallback) {
            this.resultCallback = resultCallback;
            return this;
        }

        public RequestParms build() {
            return new RequestParms(this);
        }
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeInt(this.action == null ? -1 : this.action.ordinal());
        dest.writeInt(this.resturantId);
        dest.writeTypedList(this.payMethods);
        dest.writeInt(this.payType);
        dest.writeLong(this.orderId);
        dest.writeDouble(this.payMoney);
        dest.writeString(this.refundReason);
    }

    protected RequestParms(Parcel in) {
        int tmpAction = in.readInt();
        this.action = tmpAction == -1 ? null : PayAction.values()[tmpAction];
        this.resturantId = in.readInt();
        this.payMethods = in.createTypedArrayList(PayMethod.CREATOR);
        this.payType = in.readInt();
        this.orderId = in.readLong();
        this.payMoney = in.readDouble();
        this.refundReason = in.readString();
    }

    public static final Creator<RequestParms> CREATOR = new Creator<RequestParms>() {
        @Override
        public RequestParms createFromParcel(Parcel source) {
            return new RequestParms(source);
        }

        @Override
        public RequestParms[] newArray(int size) {
            return new RequestParms[size];
        }
    };
}
