package com.gingersoft.gsa.cloud.table.mvp.model.discount;

import android.app.Activity;
import android.text.InputType;

import com.gingersoft.gsa.cloud.base.order.commodity.OrderDetail;
import com.gingersoft.gsa.cloud.base.order.discount.MultyDiscount;
import com.gingersoft.gsa.cloud.base.utils.MoneyUtil;
import com.gingersoft.gsa.cloud.database.bean.Discount;
import com.gingersoft.gsa.cloud.patterns.Strategy;
import com.gingersoft.gsa.cloud.table.mvp.ui.widget.KeyboardDialog;
import com.gingersoft.gsa.cloud.ui.widget.dialog.CommonTipDialog;
import com.jess.arms.utils.ArmsUtils;

import java.util.List;

/**
 * @author : bin
 * @create date: 2020-10-28
 * @update date: 2020-10-28
 * @description:美食折扣邏輯
 */
public class MealDiscountAction implements Strategy<Discount> {


    private Activity mContext;
    /**
     * 選中的食品或套餐組
     */
    private List<OrderDetail> mSelectMixOrderMealList;

    private List<MultyDiscount> mMultyDiscountList;

    private MealDiscountAction.OnCallBackListener mListener;

    public MealDiscountAction(Activity context, List<MultyDiscount> multyDiscountList, MealDiscountAction.OnCallBackListener listener) {
        this.mContext = context;
        this.mMultyDiscountList = multyDiscountList;
        this.mListener = listener;
    }

    public void setSelectMixOrderMealList(List<OrderDetail> selectMixOrderMealList) {
        this.mSelectMixOrderMealList = selectMixOrderMealList;
    }

    @Override
    public void action(Discount discount) {
        if (mListener == null || mSelectMixOrderMealList == null || mSelectMixOrderMealList.size() == 0) {
            return;
        }
        double mixTotalAomout = getMixTotalAomout(mSelectMixOrderMealList);
        if(discount.getType() == 0 && discount.getAmount() > mixTotalAomout){
            //金額折扣總額不能超過 食品總金額
            ArmsUtils.makeText(mContext, "折扣金額不可以超過食品總金額!");
            return;
        }
        if (discount.getMinBillAmount() > mixTotalAomout) {
            //最小賬單金額
            ArmsUtils.makeText(mContext, "小於最小賬單金額$" + discount.getMinBillAmount() + " 無法使用此折扣");
            return;
        }
        if (isOnlyOnce(discount)) {
            //此折扣只一次
            ArmsUtils.makeText(mContext, "此折扣只一次");
            return;
        }
        boolean hasExclusive = hasExclusiveDiscount();
        if (hasExclusive) {
            //有獨佔折扣
            exclusiveDiscount(discount);
            return;
        } else {
            int discountSize = getDiscountSize();
            if (discount.getExclusive() == 1 && discountSize > 0) {
                //獨佔折扣
                exclusiveDiscount(discount);
                return;
            }
        }
        switch (discount.getType()) {
            case Discount.DISCOUNT_TYPE_CUSTOM_AMOUNT:
                showCustomAmountDialog(discount, false);
                break;
            case Discount.DISCOUNT_TYPE_KEYBOARD:
                showKeyboardDialog(discount, false);
                break;
//            case Discount.DISCOUNT_TYPE_FREE_SERVICE_CHARGE:
//
//                break;
            default:
                mListener.preformAddDiscount(discount.clone());
                break;
        }
    }

    private void exclusiveDiscount(Discount discount) {
        if (mListener != null) {
            showExclusiveTipDialog(discount);
        }
    }


    /**
     * 食品是否有獨佔折扣
     *
     * @return
     */
    private boolean hasExclusiveDiscount() {
        for (OrderDetail orderMealItem : mSelectMixOrderMealList) {
            if (orderMealItem.getDiscount() != null && orderMealItem.getDiscount().getExclusive() == 1) {
                return true;
            }
        }
        return false;
    }

    /**
     * 此折扣是否只一次
     *
     * @param discount
     * @return
     */
    private boolean isOnlyOnce(Discount discount) {
        if (discount.getOnlyOnce() == 1) {
            for (OrderDetail orderMealItem : mSelectMixOrderMealList) {
                Discount item = orderMealItem.getDiscount();
                if (item != null && discount.getId().longValue() == item.getId().longValue()) {
                    return true;
                }
            }
        }
        return false;
    }


    /**
     * 顯示自定義金額彈窗
     *
     * @param discount
     */
    private void showCustomAmountDialog(Discount discount, boolean exclusiveReplace) {
        CommonTipDialog.showEditDialog(mContext, discount.getRemark(), "請輸入折扣金額", InputType.TYPE_CLASS_NUMBER | InputType.TYPE_NUMBER_FLAG_DECIMAL, new CommonTipDialog.OnEditTextListener() {
            @Override
            public void inputText(String text) {
                discount.setAmount(Double.parseDouble(text));
                if (mListener != null) {
                    if (exclusiveReplace) {
                        mListener.exclusiveRemoveOthers();
                    }
                    mListener.preformAddDiscount(discount.clone());
                }
            }
        });
    }

    /**
     * 顯示子弟你金額彈窗 鍵盤
     *
     * @param discount
     */
    private void showKeyboardDialog(Discount discount, boolean exclusiveReplace) {
        new KeyboardDialog(mContext)
                .build()
                .setOnCallBackListener(new KeyboardDialog.OnCallBackListener() {
                    @Override
                    public void onSure(String number) {
                        discount.setAmount(Double.parseDouble(number));
                        if (mListener != null) {
                            if (exclusiveReplace) {
                                mListener.exclusiveRemoveOthers();
                            }
                            mListener.preformAddDiscount(discount.clone());
                        }
                    }
                })
                .show();
    }

    /**
     * 提示有獨佔折扣是否替換
     *
     * @param discount
     */
    private void showExclusiveTipDialog(Discount discount) {
        CommonTipDialog.showDoubtDialog2(mContext, "有獨佔折扣,是否替換", new CommonTipDialog.OnCallbackListener() {
            @Override
            public void onSure() {
                switch (discount.getType()) {
                    case Discount.DISCOUNT_TYPE_CUSTOM_AMOUNT:
                        showCustomAmountDialog(discount, true);
                        break;
                    case Discount.DISCOUNT_TYPE_KEYBOARD:
                        showKeyboardDialog(discount, true);
                        break;
//                    case Discount.DISCOUNT_TYPE_FREE_SERVICE_CHARGE:
//
//                        break;
                    default:
                        mListener.exclusiveRemoveOthers();
                        mListener.preformAddDiscount(discount.clone());
                        break;
                }
            }
        });
    }

    /**
     * 獲取折扣數量
     *
     * @return
     */
    private int getDiscountSize() {
        int size = 0;
        for (OrderDetail orderMealItem : mSelectMixOrderMealList) {
            if (orderMealItem.getDiscount() != null) {
                size++;
            }
        }
        return size;
    }

    /**
     * 獲取套餐總金額
     *
     * @param mixOrderMealList
     * @return
     */
    private double getMixTotalAomout(List<OrderDetail> mixOrderMealList) {
        double mixTotalAomout = 0.0;
        for (OrderDetail orderDetail : mixOrderMealList) {
            mixTotalAomout = MoneyUtil.sum(mixTotalAomout, orderDetail.getPrice());
        }
        return mixTotalAomout;
    }


    public interface OnCallBackListener {

        void exclusiveRemoveOthers();

        void preformAddDiscount(Discount discount);
    }
}
