package com.joe.base.utils.mobile;

import android.content.Context;
import android.content.SharedPreferences;
import android.provider.Settings;
import android.telephony.TelephonyManager;

import java.io.UnsupportedEncodingException;
import java.util.UUID;

/**
 * 生成手机设备唯一UUID
 */

public class DeviceUuidFactory {
    protected static final String PREFS_FILE = "device_id.xml";
    protected static final String PREFS_DEVICE_ID = "device_id";
    protected static UUID uuid;

    public DeviceUuidFactory(Context context) {
        if (uuid == null) {
            synchronized (DeviceUuidFactory.class) {
                if (uuid == null) {
                    final SharedPreferences prefs = context.getSharedPreferences(PREFS_FILE, 0);
                    final String id = prefs.getString(PREFS_DEVICE_ID, null);
                    if (id != null) {
                        // Use the ids previously computed and stored in the prefs file
                        uuid = UUID.fromString(id);
                    } else {
                        final String androidId = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
                        // Use the Android ID unless it's broken, in which case fallback on deviceId,
                        // unless it's not available, then fallback on a random number which we store
                        // to a prefs file
                        try {
                            if (!"9774d56d682e549c".equals(androidId)) {
                                uuid = UUID.nameUUIDFromBytes(androidId.getBytes("utf8"));
                            } else {
                                final String deviceId = ((TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE)).getDeviceId();
                                uuid = deviceId != null ? UUID.nameUUIDFromBytes(deviceId.getBytes("utf8")) : UUID.randomUUID();
                            }
                        } catch (UnsupportedEncodingException e) {
                            throw new RuntimeException(e);
                        }
                        // Write the value out to the prefs file
                        prefs.edit().putString(PREFS_DEVICE_ID, uuid.toString()).commit();
                    }
                }
            }
        }
    }

    /**
     * 返回当前Android设备的唯一UUID。与所有UUID一样，这个独特的ID“非常可靠”
     * 在所有Android设备上都是独一无二的。比ANDROID_ID好多了。
     * UUID是通过使用ANDROID_ID作为基本密钥（如果适用）生成的
     * TelephonyManager.getDeviceID（）如果ANDROID_ID已知不正确，最后回落
     * 在一个随机的UUID上，如果getDeviceID（）不返回，那么它会持续到SharedPreferences
     * 可用价值。
     * 在极少数情况下，此ID可能会改变。特别是，如果设备出厂设置了新的设备ID
     * 可能会生成。另外，如果用户从Android2.2的某些错误实施中升级他们的手机
     * 到更新的，非bug的版本的Android，设备ID可能会改变。或者，如果用户卸载您的应用程序
     * 没有适当的Android ID和设备ID的设备，此ID可能会在重新安装时更改。
     * 请注意，如果代码使用TelephonyManager.getDeviceId（）返回，则生成的ID不会
     * 出厂复位后更换。需要注意的事项
     * 在直接使用ANDROID_ID时，适用于许多设备的Android2.2中的错误。**
     *
     * @http ://code.google.com/p/android/issues/detail?id=10603
     * @返回一个UUID ，可用于为大多数用途唯一标识您的设备。
     **/

    public UUID getDeviceUuid() {
        return uuid;
    }
}
