/*
 * Copyright (c) 2019-2020 gzu-liyujiang <1032694760@qq.com>
 *
 * The software is licensed under the Mulan PSL v1.
 * You can use this software according to the terms and conditions of the Mulan PSL v1.
 * You may obtain a copy of Mulan PSL v1 at:
 *     http://license.coscl.org.cn/MulanPSL
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
 * PURPOSE.
 * See the Mulan PSL v1 for more details.
 *
 */

package com.gingersoft.gsa.cloud.common.oaId.defaultImp;

import android.content.Context;
import android.os.Build;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.text.TextUtils;

import com.gingersoft.gsa.cloud.common.logan.LoganManager;

import java.security.MessageDigest;
import java.util.Locale;
import java.util.UUID;

/**
 * @作者: bin
 * @創建時間: 2020-12-25 14:31
 * @更新時間: 2020-12-25 14:31
 * @描述: OAID 獲取不到後的其他備選方案
 * <p>
 * 方案1:相较于前两种方案，限制性较小，只要硬件信息不变结果就不变。而且该方案可自行定制组合。 TODO 硬件标识对Android 10来说都不可取了
 * 方案2:是很多软件采用的方案，因很少有人删除SD卡文件；但需要注意权限；
 * 方案3:局限性较大，不建议使用；
 */
public class DefaultDeviceIdUtils {

    private final String TAG = getClass().getSimpleName();
    /**
     * 方案1：imei + android_id + serial + 硬件uuid（自生成）
     * 如果，又想唯一，又不想因用户的删除而重新生成UUID，该怎么做呢？
     * <p>
     * 不依赖随机的UUID，咱们根据硬件标识来创建唯一的数据；
     * <p>
     * 我们可以将多个可获得的硬件标识拼接起来（尽可能不依赖权限），最大程度上降低重复性。
     * 以imei、android_id、serial为例，如果能取到值，每个数据几乎可以代表唯一。
     * 如果这些数据都能获取到，拼起来的数据重复性降到极低（UUID也存在重复性，重复性极低而已）
     * <p>
     * AndroidId : 如：df176fbb152ddce,无需权限,极个别设备获取不到数据或得到错误数据；
     * serial：如：LKX7N18328000931,无需权限,极个别设备获取不到数据；
     * IMEI : 如：23b12e30ec8a2f17，需要权限；
     * Mac: 如：6e:a5:....需要权限，高版本手机获得数据均为 02:00.....（不可使用）
     * Build.BOARD  如：BLA  主板名称,无需权限,同型号设备相同
     * Build.BRAND  如：HUAWEI  厂商名称,无需权限,同型号设备相同
     * Build.HARDWARE  如：kirin970  硬件名称,无需权限,同型号设备相同
     * Build......更多硬件信息，略
     * <p>
     * 分析了这么多硬件标识，我们就使用imei + android_id + serial + 硬件UUID（使用Build属性生成，如果硬件信息不变，则UUID值不变）。这是我们项目的实际方案，大家也可根据自己的需要自由组合硬件标识。
     * 那么，问题又来了，不同设备的硬件标识长度不同，拼接处理的DeviceId字符串长度不同，怎么才能统一长度呢？
     * 也很简单，我们先拼接好DeviceId数据，取其SHA1值，再转16进制即可（统一40位长度）
     */
    private static final String scheme_1 = "iasu";

    /**
     * 獲取設備ID
     *
     * @param context
     * @return
     */
    public static final String getDefaultDeviceIds(Context context) {
        //第一方案獲取ID失敗
        String deviceId = getDeviceId(context);
        if (deviceId == null) {
            String defaultDeviceId = UUID.randomUUID().toString().replace("-", "");
            //使用第二方案獲取（用戶未給與SD卡權限）
            deviceId = SDDeviceId.getSDDeviceId(defaultDeviceId);
            if (TextUtils.isEmpty(deviceId)) {
                //使用第三方案獲取
                deviceId = SPDeviceId.getSPDeviceId(defaultDeviceId);
            }
        }
        LoganManager.w_code("getDefaultDeviceIds","getDefaultDeviceIds = "+deviceId);
        return deviceId;
    }


    /**
     * 获得设备硬件标识
     *
     * @param context 上下文
     * @return 设备硬件标识
     */
    public static String getDeviceId(Context context) {
        StringBuilder sbDeviceId = new StringBuilder();

        //获得设备默认IMEI（>=6.0 需要ReadPhoneState权限）
        String imei = getIMEI(context);
        //获得AndroidId（无需权限）
        String androidid = getAndroidId(context);
        //获得设备序列号（无需权限）
        String serial = getSERIAL();
        //获得硬件uuid（根据硬件相关属性，生成uuid）（无需权限）
        String uuid = getDeviceUUID().replace("-", "");

        //追加imei
        if (imei != null && imei.length() > 0) {
            sbDeviceId.append(imei);
            sbDeviceId.append("|");
        }
        //追加androidid
        if (androidid != null && androidid.length() > 0) {
            sbDeviceId.append(androidid);
            sbDeviceId.append("|");
        }
        //追加serial
        if (serial != null && serial.length() > 0) {
            sbDeviceId.append(serial);
            sbDeviceId.append("|");
        }
        //追加硬件uuid
        if (uuid != null && uuid.length() > 0) {
            sbDeviceId.append(uuid);
        }

        //生成SHA1，统一DeviceId长度
        if (sbDeviceId.length() > 0) {
            try {
                byte[] hash = getHashByString(sbDeviceId.toString());
                String sha1 = bytesToHex(hash);
                if (sha1 != null && sha1.length() > 0) {
                    //返回最终的DeviceId
                    return scheme_1 + "_" + sha1;
                }
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }

        //如果以上硬件标识数据均无法获得，
        //则DeviceId默认使用系统随机数，这样保证DeviceId不为空
//        return UUID.randomUUID().toString().replace("-", "");
        return null;
    }

    //需要获得READ_PHONE_STATE权限，>=6.0，默认返回null
    private static String getIMEI(Context context) {
        try {
            TelephonyManager tm = (TelephonyManager)
                    context.getSystemService(Context.TELEPHONY_SERVICE);
            return tm.getDeviceId();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return "";
    }

    /**
     * 获得设备的AndroidId
     *
     * @param context 上下文
     * @return 设备的AndroidId
     */
    private static String getAndroidId(Context context) {
        try {
            return Settings.Secure.getString(context.getContentResolver(),
                    Settings.Secure.ANDROID_ID);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return "";
    }

    /**
     * 获得设备序列号（如：WTK7N16923005607）, 个别设备无法获取
     *
     * @return 设备序列号
     */
    private static String getSERIAL() {
        try {
            return Build.SERIAL;
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return "";
    }

    /**
     * 获得设备硬件uuid
     * 使用硬件信息，计算出一个随机数
     *
     * @return 设备硬件uuid
     */
    private static String getDeviceUUID() {
        try {
            String dev = "3883756" +
                    Build.BOARD.length() % 10 +
                    Build.BRAND.length() % 10 +
                    Build.DEVICE.length() % 10 +
                    Build.HARDWARE.length() % 10 +
                    Build.ID.length() % 10 +
                    Build.MODEL.length() % 10 +
                    Build.PRODUCT.length() % 10 +
                    Build.SERIAL.length() % 10;
            return new UUID(dev.hashCode(),
                    Build.SERIAL.hashCode()).toString();
        } catch (Exception ex) {
            ex.printStackTrace();
            return "";
        }
    }

    /**
     * 取SHA1
     *
     * @param data 数据
     * @return 对应的hash值
     */
    private static byte[] getHashByString(String data) {
        try {
            MessageDigest messageDigest = MessageDigest.getInstance("SHA1");
            messageDigest.reset();
            messageDigest.update(data.getBytes("UTF-8"));
            return messageDigest.digest();
        } catch (Exception e) {
            return "".getBytes();
        }
    }

    /**
     * 转16进制字符串
     *
     * @param data 数据
     * @return 16进制字符串
     */
    private static String bytesToHex(byte[] data) {
        StringBuilder sb = new StringBuilder();
        String stmp;
        for (int n = 0; n < data.length; n++) {
            stmp = (Integer.toHexString(data[n] & 0xFF));
            if (stmp.length() == 1)
                sb.append("0");
            sb.append(stmp);
        }
        return sb.toString().toUpperCase(Locale.CHINA);
    }


}
