package com.gingersoft.gsa.cloud.ui.view;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.gingersoft.gsa.cloud.common.utils.time.TimeUtils;
import com.gingersoft.gsa.cloud.ui.R;
import com.gingersoft.gsa.cloud.ui.utils.TimePickerUtils;

import java.text.ParseException;
import java.util.Calendar;

public class StartEndTimePickerView extends FrameLayout implements View.OnClickListener {

    private TextView tvStartTime;
    private TriangleView startTriangleView;
    private TextView tvEndTime;
    private TriangleView endTriangleView;

    private onUpdateTimePicker onUpdateTimePicker;

    private int mTextColor;
    private float mTextSize;
    private float mImgWidth;
    private float mImgHeight;
    private Drawable mArrowRes;
    private boolean showStartTime;
    private boolean showEndTime;

    public StartEndTimePickerView(@NonNull Context context) {
        this(context, null, -1);
    }

    public StartEndTimePickerView(@NonNull Context context, onUpdateTimePicker onUpdateTimePicker) {
        this(context);
        this.onUpdateTimePicker = onUpdateTimePicker;
    }

    public StartEndTimePickerView(@NonNull Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, -1);
    }

    public StartEndTimePickerView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context, attrs);
    }

    public void setOnUpdateTimePicker(StartEndTimePickerView.onUpdateTimePicker onUpdateTimePicker) {
        this.onUpdateTimePicker = onUpdateTimePicker;
    }

    private void init(Context context, AttributeSet attrs) {
        initTypedArray(context, attrs);
        View view = View.inflate(context, R.layout.layout_start_end_time_picker, null);
        tvStartTime = view.findViewById(R.id.tv_start_time);
        startTriangleView = view.findViewById(R.id.iv_start_time_triangle);
        tvEndTime = view.findViewById(R.id.tv_end_time);
        endTriangleView = view.findViewById(R.id.iv_end_time_triangle);

        tvStartTime.setTextColor(mTextColor);
        tvEndTime.setTextColor(mTextColor);
        tvStartTime.setTextSize(TypedValue.COMPLEX_UNIT_PX, mTextSize);
        tvEndTime.setTextSize(TypedValue.COMPLEX_UNIT_PX, mTextSize);
        TextView tvInterval = view.findViewById(R.id.tv_time_interval);
        tvInterval.setTextSize(TypedValue.COMPLEX_UNIT_PX, mTextSize);

        tvStartTime.setVisibility(showStartTime ? VISIBLE : GONE);
        tvEndTime.setVisibility(showEndTime ? VISIBLE : GONE);
        if (mArrowRes != null) {
            startTriangleView.setImageDrawable(mArrowRes);
            endTriangleView.setImageDrawable(mArrowRes);
        }
        LinearLayout.LayoutParams layoutParams = (LinearLayout.LayoutParams) startTriangleView.getLayoutParams();
        layoutParams.width = (int) mImgWidth;
        layoutParams.height = (int) mImgHeight;
        startTriangleView.setLayoutParams(layoutParams);
        endTriangleView.setLayoutParams(layoutParams);

        tvStartTime.setText(TimeUtils.getOldDate(-7));
        tvEndTime.setText(TimeUtils.getOldDate(0));

        tvStartTime.setOnClickListener(this);
        startTriangleView.setOnClickListener(this);
        tvEndTime.setOnClickListener(this);
        endTriangleView.setOnClickListener(this);
        updateTime();
        this.addView(view);
    }

    private void initTypedArray(Context context, AttributeSet attrs) {
        TypedArray typedArray = context.obtainStyledAttributes(attrs, R.styleable.StartEndTimePickerView);
        mTextColor = typedArray.getColor(R.styleable.StartEndTimePickerView_textColor, ContextCompat.getColor(context, R.color.color_18));
        mTextSize = typedArray.getDimension(R.styleable.StartEndTimePickerView_textSize, getResources().getDimension(R.dimen.dp_10));
        mImgWidth = typedArray.getDimension(R.styleable.StartEndTimePickerView_imgWidth, getResources().getDimension(R.dimen.dp_8));
        mImgHeight = typedArray.getDimension(R.styleable.StartEndTimePickerView_imgHeight, getResources().getDimension(R.dimen.dp_5));
        mArrowRes = typedArray.getDrawable(R.styleable.StartEndTimePickerView_arrowRes);
        showStartTime = typedArray.getBoolean(R.styleable.StartEndTimePickerView_showStartTime, true);
        showEndTime = typedArray.getBoolean(R.styleable.StartEndTimePickerView_showEndTime, true);
        typedArray.recycle();
    }

    @Override
    public void onClick(View v) {
        int viewId = v.getId();
        try {
            String time;
            Calendar startTime = Calendar.getInstance();
            startTime.set(Calendar.DAY_OF_YEAR, startTime.getActualMinimum(Calendar.DAY_OF_YEAR));//今年第一天

            Calendar endTime = Calendar.getInstance();

            if (viewId == R.id.tv_start_time || viewId == R.id.iv_start_time_triangle) {
                startTriangleView.toggle();
                time = tvStartTime.getText().toString();

                endTime.setTime(TimeUtils.DATE_FORMAT_DATE.parse(tvEndTime.getText().toString()));
            } else {
                endTriangleView.toggle();
                time = tvEndTime.getText().toString();
                startTime.setTime(TimeUtils.DATE_FORMAT_DATE.parse(tvStartTime.getText().toString()));
            }

            Calendar cal = Calendar.getInstance();
            try {
                //設置默認時間為當前的起止時間
                cal.setTime(TimeUtils.DATE_FORMAT_DATE.parse(time));
            } catch (ParseException e) {
                e.printStackTrace();
            }
            //打開時間選擇器
            TimePickerUtils.showReportTimePicker(v.getContext(), startTime, endTime, cal, (date, v1) ->
            {
                if (viewId == R.id.tv_start_time || viewId == R.id.iv_start_time_triangle) {
                    tvStartTime.setText(TimeUtils.DATE_FORMAT_DATE.format(date));
                } else {
                    tvEndTime.setText(TimeUtils.DATE_FORMAT_DATE.format(date));
                }
                updateTime();
            }).setOnDismissListener(o -> {
                if (startTriangleView.isOpen()) {
                    startTriangleView.toggle();
                } else if (endTriangleView.isOpen()) {
                    endTriangleView.toggle();
                }
            });
        } catch (ParseException e) {
            e.printStackTrace();
        }
    }

    private void updateTime() {
        if (onUpdateTimePicker != null) {
            onUpdateTimePicker.onUpdateTime(tvStartTime.getText().toString(), tvEndTime.getText().toString());
        }
    }


    public interface onUpdateTimePicker {
        void onUpdateTime(String startTime, String endTime);
    }

    public StartEndTimePickerView setStartTime(String startTime) {
        tvStartTime.setText(startTime);
        return this;
    }

    public StartEndTimePickerView setEndTime(String endTime) {
        tvEndTime.setText(endTime);
        return this;
    }

    public String getStartTime() {
        if (tvStartTime != null) {
            return tvStartTime.getText().toString();
        }
        return TimeUtils.getOldDate(-7);
    }

    public String getEndTime() {
        if (tvEndTime != null) {
            return tvEndTime.getText().toString();
        }
        return TimeUtils.getOldDate(0);
    }

}
