package com.gingersoft.gsa.cloud.table.mvp.action.discount;

import android.app.Activity;
import android.text.InputType;

import com.gingersoft.gsa.cloud.common.patterns.Strategy;
import com.gingersoft.gsa.cloud.database.bean.Discount;
import com.gingersoft.gsa.cloud.order.discount.NomalDiscount;
import com.gingersoft.gsa.cloud.ui.widget.dialog.CommonTipDialog;
import com.gingersoft.gsa.cloud.ui.widget.dialog.KeyboardDialog;
import com.jess.arms.utils.ArmsUtils;

import java.util.List;

import lombok.Builder;
import lombok.Setter;

/**
 * @author : bin
 * @create date: 2020-10-27
 * @update date: 2020-10-27
 * @description:賬單折扣邏輯
 */
@Builder
@Setter
public class BillDiscountAction implements Strategy<Discount> {

    private Activity context;

    /**
     * 賬單折扣列表
     */
    private List<NomalDiscount> nomalDiscountList;

    private double wholeAmount;

    private OnCallBackListener listener;

    private String cashStr ;

    @Override
    public void action(Discount discount) {
        if (listener == null) {
            return;
        }
        if (discount.getMinBillAmount() > wholeAmount) {
            //小於最小賬單金額
            ArmsUtils.makeText(context, "小於最小賬單金額" + cashStr + discount.getMinBillAmount() + " 無法使用此折扣");
            return;
        }
        if (isOnlyOnce(discount)) {
            //此折扣只一次
            ArmsUtils.makeText(context, "此折扣只一次");
            return;
        }
        boolean hasExclusive = hasExclusiveDiscount();
        if (hasExclusive) {
            //有獨佔折扣
            exclusiveDiscount(discount);
            return;
        } else {
            if (nomalDiscountList != null && discount.getExclusive() == 1 && nomalDiscountList.size() > 0) {
                //獨佔折扣
                exclusiveDiscount(discount);
                return;
            }
        }
        switch (discount.getType()) {
            case Discount.DISCOUNT_TYPE_CUSTOM_AMOUNT:
                showCustomAmountDialog(discount, false);
                break;
            case Discount.DISCOUNT_TYPE_KEYBOARD:
                showKeyboardDialog(discount, false);
                break;
//            case Discount.DISCOUNT_TYPE_FREE_SERVICE_CHARGE:
//
//                break;
            default:
                listener.preformAddDiscount(discount.clone());
                break;
        }
    }

    @Override
    public void destroy() {

    }

    private void exclusiveDiscount(Discount discount) {
        if (listener != null) {
            showExclusiveTipDialog(discount);
        }
    }

    /**
     * 是否有獨佔折扣
     *
     * @return
     */
    private boolean hasExclusiveDiscount() {
        for (NomalDiscount nomalDiscount : nomalDiscountList) {
            if (nomalDiscount.getDiscount().getExclusive() == 1) {
                return true;
            }
        }
        return false;
    }

    /**
     * 此折扣是否只一次
     *
     * @param discount
     * @return
     */
    private boolean isOnlyOnce(Discount discount) {
        if (discount.getOnlyOnce() == 1) {
            for (NomalDiscount nomalDiscount : nomalDiscountList) {
                Discount item = nomalDiscount.getDiscount();
                if (discount.getId().longValue() == item.getId().longValue()) {
                    return true;
                }
            }
        }
        return false;
    }


    /**
     * 顯示自定義金額彈窗
     *
     * @param discount
     */
    private void showCustomAmountDialog(Discount discount, boolean exclusiveReplace) {
        CommonTipDialog.showEditDialog(context, discount.getRemark(), "請輸入折扣金額", InputType.TYPE_CLASS_NUMBER | InputType.TYPE_NUMBER_FLAG_DECIMAL, new CommonTipDialog.OnEditTextListener() {
            @Override
            public void inputText(String text) {
                discount.setAmount(Double.parseDouble(text));
                if (listener != null) {
                    if (exclusiveReplace) {
                        listener.exclusiveRemoveOthers(nomalDiscountList);
                    }
                    listener.preformAddDiscount(discount.clone());
                }
            }
        });
    }

    /**
     * 顯示子弟你金額彈窗 鍵盤
     *
     * @param discount
     */
    private void showKeyboardDialog(Discount discount, boolean exclusiveReplace) {
        new KeyboardDialog(context)
                .build()
                .setOnCallBackListener(new KeyboardDialog.OnCallBackListener() {
                    @Override
                    public void onSure(String number) {
                        discount.setAmount(Double.parseDouble(number));
                        if (listener != null) {
                            if (exclusiveReplace) {
                                listener.exclusiveRemoveOthers(nomalDiscountList);
                            }
                            listener.preformAddDiscount(discount.clone());
                        }
                    }
                })
                .show();
    }

    /**
     * 提示有獨佔折扣是否替換
     *
     * @param discount
     */
    private void showExclusiveTipDialog(Discount discount) {
        CommonTipDialog.showDoubtDialog2(context, "有獨佔折扣,是否替換", new CommonTipDialog.OnCallbackListener() {
            @Override
            public void onSure() {
                switch (discount.getType()) {
                    case Discount.DISCOUNT_TYPE_CUSTOM_AMOUNT:
                        showCustomAmountDialog(discount, true);
                        break;
                    case Discount.DISCOUNT_TYPE_KEYBOARD:
                        showKeyboardDialog(discount, true);
                        break;
//                    case Discount.DISCOUNT_TYPE_FREE_SERVICE_CHARGE:
//
//                        break;
                    default:
                        listener.exclusiveRemoveOthers(nomalDiscountList);
                        listener.preformAddDiscount(discount.clone());
                        break;
                }
            }
        });
    }

    public interface OnCallBackListener {

        void exclusiveRemoveOthers(List<NomalDiscount> nomalDiscountList);

        void preformAddDiscount(Discount discount);
    }

}
