package com.gingersoft.gsa.cloud.common.function.click;

import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import com.gingersoft.gsa.cloud.common.R;
import com.gingersoft.gsa.cloud.common.logan.LoganManager;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;

import java.util.Calendar;

import butterknife.internal.DebouncingOnClickListener;

/**
 * 全局點擊事件AOP
 *
 * @author admin
 */
@Aspect
public class BaseClickAspect {
    static int TIME_TAG = R.id.click_time;

    protected final String TAG = this.getClass().getSimpleName();

    /**
     * 方法切入点
     */
    @Pointcut("execution(* android.view.View.OnClickListener.onClick(..))")
    public void onClickMethod() {
    }

    /**
     * 在连接点进行方法替换
     *
     * @param joinPoint
     * @throws Throwable
     */
    @Around("onClickMethod()")
    public void doClickMethod(ProceedingJoinPoint joinPoint) throws Throwable {
        View view = null;
        Class<?> declaringType = joinPoint.getSignature().getDeclaringType();
        if (declaringType == DebouncingOnClickListener.class) {
            // 是butterknife註解，不攔截。因為butterknife底層實現也是OnClickListener
            // 所以第一次是butterknife，第二次才是真正的執行
            joinPoint.proceed();//执行原方法
            return;
        }
        Object[] args = joinPoint.getArgs();
        for (Object arg : args) {
            if (arg instanceof View) {
                view = (View) arg;
            }
            if (view != null) {
                getViewInfo(view);
                Object tag = view.getTag(TIME_TAG);
                long lastClickTime = ((tag != null) ? (long) tag : 0);
                long currentTime = Calendar.getInstance().getTimeInMillis();
                if (currentTime - lastClickTime > 500) {//过滤掉500毫秒内的连续点击
                    view.setTag(TIME_TAG, currentTime);
                    joinPoint.proceed();//执行原方法
                }
            }
        }
    }

    private void getViewInfo(View view) {
        if (recordText(view)) {
            return;
        } else if (view instanceof ViewGroup) {
            ViewGroup viewGroup = (ViewGroup) view;
            int childCount = viewGroup.getChildCount();
            for (int i = 0; i < childCount; i++) {
                if (recordText(viewGroup.getChildAt(i))) {
                    return;
                }
            }
            //沒有子view是textview的。就查找子view有沒有有id的
            for (int i = 0; i < childCount; i++) {
                if (recordId(viewGroup.getChildAt(i))) {
                    return;
                }
            }
        }
        recordId(view);
    }

    private boolean recordText(View view) {
        if (view instanceof TextView) {
            TextView textView = (TextView) view;
            LoganManager.w_global(TAG, LoganManager.EVENT_CLICK + textView.getText());
            return true;
        }
        return false;
    }

    private boolean recordId(View view) {
        if (view.getId() != View.NO_ID) {
            String resourceEntryName = view.getContext().getResources().getResourceEntryName(view.getId());
            LoganManager.w_global(TAG, LoganManager.EVENT_CLICK + resourceEntryName);
            return true;
        }
        return false;
    }
}
