package com.joe.print.mvp.print.utils;

import android.content.Intent;
import android.text.TextUtils;
import android.util.Xml;

import com.gingersoft.gsa.cloud.app.GsaCloudApplication;
import com.gingersoft.gsa.cloud.common.utils.okhttpUtils.OkHttp3Utils;
import com.joe.print.mvp.print.PrintListener;
import com.joe.print.mvp.ui.activity.PrintActivity;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.ConnectException;
import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.TimeoutException;

import io.reactivex.Observable;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.schedulers.Schedulers;

public class BBposPrint {

    /**
     * RESULT_CANCELED : 0
     * RESULT_OK : -1
     * RESULT_FAILED : -2
     * RESULT_BUSY : -3
     * RESULT_DEV_ERR : -4
     * RESULT_NET_ERR : -5
     * RESULT_UNKNOWN : -6
     * RESULT_UNKNOWN_MERCHANT : -7
     * RESULT_INPUT_INVALID : -8
     */

    private static final String RESULT_OK = "RESULT_OK";        //成功
    private static final String RESULT_FAILED = "RESULT_FAILED";//失敗
    private static final String RESULT_CANCELED = "RESULT_CANCELED";//取消
    private static final String RESULT_BUSY = "RESULT_BUSY"; //繁忙
    private static final String RESULT_DEV_ERR = "RESULT_DEV_ERR";//开发错误
    private static final String RESULT_NET_ERR = "RESULT_NET_ERR";//網絡錯誤
    private static final String RESULT_UNKNOWN = "RESULT_UNKNOWN";//未知錯誤
    private static final String RESULT_UNKNOWN_MERCHANT = "RESULT_UNKNOWN_MERCHANT";//未知的商家
    private static final String RESULT_INPUT_INVALID = "RESULT_INPUT_INVALID";//输入无效

    private static final String HTML_FORMAT_ERR = "HTML is not complete, please make sure special characters are escaped";
    /**
     * 有打印多張的情況 用隊列來維護
     */
    private Queue<String> printXMLQueues = new LinkedList<>();
    /**
     * 打印成功數量
     */
    private int printSuccesCount;
    /**
     * 打印異常數量
     */
    private int printErrorCount;
    /**
     * 最大遞歸數量，超過這個數不管失敗幾次都直接退出打印
     */
    private int maxRecursionCount = 3;

    private Observable<String> stringObservable;

    /**
     * 递归打印
     *
     * @param printDatas 打印数据
     * @param listener   回调监听
     * @param recursion  是否递归从队列中取打印数据
     */
    public void print(String[] printDatas, PrintListener listener, boolean recursion) {
        if (!recursion) {
            if (printXMLQueues.size() > 0) {
                printXMLQueues.clear();
            }
            //将打印数据添加到队列中
            for (String printData : printDatas) {
                printXMLQueues.offer(printData);
                printSuccesCount++;
            }
        }
        //poll從隊列中取打印數據 取不到會返回null
        String printContent = printXMLQueues.peek();
        //超過最大遞歸數或已經打印完畢退出打印
        if (maxRecursionCount < 1 || TextUtils.isEmpty(printContent)) {
            backPrintActivity();
            return;
        }
        stringObservable = OkHttp3Utils.get("http://localhost:8080/pos?transactionType=PRINT&isExternal=false&apiVersion=21&printData=" + printContent);
        stringObservable.subscribeOn(Schedulers.io());
        stringObservable.subscribeOn(AndroidSchedulers.mainThread());
        stringObservable.observeOn(AndroidSchedulers.mainThread());
        stringObservable.subscribe(new Observer<String>() {

            @Override
            public void onSubscribe(Disposable d) {
            }

            @Override
            public void onNext(String s) {
                try {
                    String[] xmlContents = analysisPrintResult(s, printDatas, listener, recursion);
                    if (xmlContents.length == 3) {
                        String status = xmlContents[0];
                        String resultCode = xmlContents[1];
                        String errorMessage = xmlContents[2];//HTML is not complete, please make sure special characters are escaped
                        if (!TextUtils.isEmpty(resultCode)) {
                            switch (resultCode) {
                                case RESULT_OK:
                                    printSuccesCount--;
                                    break;
                                case RESULT_CANCELED:
                                case RESULT_FAILED:
                                case RESULT_BUSY:
                                case RESULT_DEV_ERR:
                                case RESULT_NET_ERR:
                                case RESULT_UNKNOWN:
                                case RESULT_UNKNOWN_MERCHANT:
                                case RESULT_INPUT_INVALID:
                                    printErrorCount++;
                                    break;
                                default:
                                    break;
                            }
                        } else {
                            printErrorCount++;
                            if (errorMessage.equals(HTML_FORMAT_ERR)) {
                                listener.printFailure("內容格式錯誤,無法打印!");
                            } else {
                                listener.printFailure(errorMessage);
                            }
                        }
                    } else {
                        //TODO 解析xml問題
                        printErrorCount++;
                    }
                    printNext(printDatas, listener);
                } catch (XmlPullParserException | IOException e) {
                    e.printStackTrace();
                    if (s.contains("Failed to connect")) {
                        listener.printFailure("無法連接打印，請檢查WisePay是否打開");
                    } else if (s.contains("HTML is not complete")) {
                        listener.printFailure("內容格式錯誤,無法打印!");
                    } else if (s.contains("Please login first")) {
                        login(printDatas, listener, recursion);
                    }
                    backPrintActivity();
                }
            }

            //HTML不完整，请确保转义特殊字符
            @Override
            public void onError(Throwable e) {
                e.printStackTrace();
                if (e instanceof ConnectException) {
                    listener.printFailure("無法連接打印，請檢查WisePay是否打開");
                } else if (e instanceof TimeoutException) {
                    backPrintActivity();
                }
                printErrorCount++;
                maxRecursionCount--;
            }

            @Override
            public void onComplete() {
                maxRecursionCount--;
                if ((printSuccesCount + printErrorCount) == printXMLQueues.size()) {
                    //打印完畢
                    if (printErrorCount == 0) {
                        //部分打印失敗
                        listener.printFailure("打印成功" + printSuccesCount + "張,失敗" + printErrorCount + "張");
                    } else {
                        //全部打印成功
                        listener.printSuccess();
                    }
                    return;
                }
            }
        });
    }

    /**
     * 登录bbpos
     *
     * @param printDatas
     * @param listener
     * @param recursion
     */
    public void login(String[] printDatas, PrintListener listener, boolean recursion) {
        OkHttp3Utils.get("http://localhost:8080/pos?UserName=gs01&Password=88888888&transactionType=LOGIN&isExternal=true&apiVersion=21")
                .subscribeOn(Schedulers.io())
                .subscribeOn(AndroidSchedulers.mainThread())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Observer<String>() {
                    @Override
                    public void onSubscribe(Disposable d) {
                    }

                    @Override
                    public void onNext(String s) {
                        //登录成功 开始打印
                        print(printDatas, listener, recursion);
                    }

                    @Override
                    public void onError(Throwable e) {
                        e.printStackTrace();
                        if (e instanceof ConnectException) {
                            listener.printFailure("登錄失敗,無法打印!");
                        }
                    }

                    @Override
                    public void onComplete() {
                    }
                });
    }

    private void backPrintActivity() {
        closePrintObservable();
        Intent intent = new Intent(GsaCloudApplication.getAppContext(), PrintActivity.class);
        intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        intent.addFlags(Intent.FLAG_ACTIVITY_RESET_TASK_IF_NEEDED);
        intent.addFlags(Intent.FLAG_ACTIVITY_PREVIOUS_IS_TOP);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        GsaCloudApplication.getAppContext().startActivity(intent);
    }

    //    private void analysisPrintResult(String s, String[] printDatas, WiseposBean wiseposBean, PrintListener listener, boolean recursion) throws XmlPullParserException, IOException {
    private String[] analysisPrintResult(String s, String[] printDatas, PrintListener listener, boolean recursion) throws XmlPullParserException, IOException {
        XmlPullParser xmlPullParser = Xml.newPullParser();
        xmlPullParser.setInput(new ByteArrayInputStream(s.getBytes()), "utf-8");
        int eventType = xmlPullParser.getEventType();
        String[] xmlContents = new String[3];
        while (eventType != XmlPullParser.END_DOCUMENT) {
            switch (eventType) {
                case XmlPullParser.START_DOCUMENT:
                    //開始解析
                    break;
                case XmlPullParser.START_TAG:
                    //解析到某一個節點
                    String xmlName = xmlPullParser.getName();
                    switch (xmlName) {
                        case "status":
                            xmlContents[xmlContents.length] = xmlPullParser.nextText();
                            break;
                        case "resultCode":
                            xmlContents[xmlContents.length] = xmlPullParser.nextText();
                            break;
                        case "ErrorMessage":
//                            if (xmlPullParser.nextText().equals("Please login first")) {
////                                    listener.printFailure("請先登錄WisePay");
//                                closePrintObservable();
//                                login(printDatas, listener, recursion);
//                            } else {
//                                listener.printFailure("" + xmlPullParser.nextText());
//                            }
                            xmlContents[xmlContents.length] = xmlPullParser.nextText();
                            break;
                        case "requestType":
//                            wiseposBean.setRequestType(xmlPullParser.nextText());
                            break;
                        default:
                            break;
                    }
                    break;
                case XmlPullParser.END_TAG:

                    break;
                default:
                    break;
            }
            eventType = xmlPullParser.next();
        }
        return xmlContents;
    }

    private void printNext(String[] printDatas, PrintListener listener) {
        //打印完一張就移除
        printXMLQueues.remove();
        //递归进行打印
        print(printDatas, listener, true);
    }

    private void closePrintObservable() {
        if (stringObservable != null && !stringObservable.subscribe().isDisposed()) {
            stringObservable.subscribe().dispose();
            stringObservable = null;
        }
    }
}
