package com.gingersoft.supply_chain.mvp.presenter;

import android.app.Application;

import com.gingersoft.gsa.cloud.common.bean.BaseResult;
import com.gingersoft.gsa.cloud.common.constans.AppConstant;
import com.gingersoft.gsa.cloud.common.utils.CollectionUtils;
import com.gingersoft.gsa.cloud.common.utils.JsonUtils;
import com.gingersoft.gsa.cloud.common.utils.MoneyUtil;
import com.gingersoft.gsa.cloud.common.utils.gson.GsonUtils;
import com.gingersoft.gsa.cloud.common.utils.other.TextUtil;
import com.gingersoft.gsa.cloud.common.utils.view.BitmapUtil;
import com.gingersoft.supply_chain.R;
import com.gingersoft.supply_chain.mvp.bean.DeputyUnitBean;
import com.gingersoft.supply_chain.mvp.bean.NoIsFoodSnTipBean;
import com.gingersoft.supply_chain.mvp.bean.OrderWareHouseBean;
import com.gingersoft.supply_chain.mvp.bean.PurchaseFoodBean;
import com.gingersoft.supply_chain.mvp.bean.PurchaseFoodEncodeSn;
import com.gingersoft.supply_chain.mvp.bean.PurchaseOrderDetailsBean;
import com.gingersoft.supply_chain.mvp.bean.WareHousingUnitBean;
import com.gingersoft.supply_chain.mvp.content.PurchaseConstant;
import com.gingersoft.supply_chain.mvp.content.SupplyShoppingCart;
import com.gingersoft.supply_chain.mvp.contract.OrderDetailsContract;
import com.jess.arms.di.scope.FragmentScope;
import com.jess.arms.http.imageloader.ImageLoader;
import com.jess.arms.integration.AppManager;
import com.jess.arms.mvp.BasePresenter;
import com.jess.arms.utils.RxLifecycleUtils;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;

import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.annotations.NonNull;
import io.reactivex.schedulers.Schedulers;
import me.jessyan.rxerrorhandler.core.RxErrorHandler;
import me.jessyan.rxerrorhandler.handler.ErrorHandleSubscriber;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.RequestBody;


/**
 * ================================================
 * Description:
 * <p>
 * Created by MVPArmsTemplate on 12/22/2020 16:57
 * <a href="mailto:jess.yan.effort@gmail.com">Contact me</a>
 * <a href="https://github.com/JessYanCoding">Follow me</a>
 * <a href="https://github.com/JessYanCoding/MVPArms">Star me</a>
 * <a href="https://github.com/JessYanCoding/MVPArms/wiki">See me</a>
 * <a href="https://github.com/JessYanCoding/MVPArmsTemplate">模版请保持更新</a>
 * ================================================
 */
@FragmentScope
public class OrderDetailsPresenter extends BasePresenter<OrderDetailsContract.Model, OrderDetailsContract.View> {
    @Inject
    RxErrorHandler mErrorHandler;
    @Inject
    Application mApplication;
    @Inject
    ImageLoader mImageLoader;
    @Inject
    AppManager mAppManager;

    private int orderId;

    private int supplierId;
    /**
     * 所上傳的圖片路徑集合
     */
    private List<Object> imgs;
    /**
     * 逗號分隔符
     */
    private final String COMMA_SEPARATED = ",";
    /**
     * 訂單狀態
     */
    public int orderState;

    @Inject
    public OrderDetailsPresenter(OrderDetailsContract.Model model, OrderDetailsContract.View rootView) {
        super(model, rootView);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        this.mErrorHandler = null;
        this.mAppManager = null;
        this.mImageLoader = null;
        this.mApplication = null;
    }

    /**
     * 跟劇訂單ID獲取訂單詳情
     *
     * @param orderId 訂單ID
     */
    public void getOrderDetailsById(int orderId) {
        mModel.getOrderDetails(orderId)
                .subscribeOn(Schedulers.io())
                .doOnSubscribe(disposable -> mRootView.showLoading(AppConstant.GET_INFO_LOADING))
//                .subscribeOn(AndroidSchedulers.mainThread())
                .observeOn(AndroidSchedulers.mainThread())
                .doAfterTerminate(() -> mRootView.hideLoading())
                .compose(RxLifecycleUtils.bindToLifecycle(mRootView))
                .subscribe(new ErrorHandleSubscriber<BaseResult>(mErrorHandler) {
                    @Override
                    public void onNext(BaseResult baseResult) {
                        if (baseResult.isSuccess()) {
                            PurchaseOrderDetailsBean orderDetailsBean = GsonUtils.GsonToBean(GsonUtils.GsonString(baseResult.getData()), PurchaseOrderDetailsBean.class);
                            if (orderDetailsBean != null) {
                                OrderDetailsPresenter.this.orderId = orderId;
                                try {
                                    supplierId = orderDetailsBean.getSupplierId();
                                    transformationWareHouse(orderDetailsBean);
                                    orderState = orderDetailsBean.getStatus();
                                    //加載圖片
                                    loadImages(orderDetailsBean.getFiles());
                                    mRootView.loadOrderDetails(orderDetailsBean);
                                } catch (NullPointerException e) {
                                    mRootView.showMessage(e.getMessage());
                                    mRootView.killMyself();
                                }
                            } else {
                                mRootView.showMessage(AppConstant.GET_INFO_ERROR);
                            }
                        } else if (TextUtil.isNotEmptyOrNullOrUndefined(baseResult.getErrMsg())) {
                            mRootView.showMessage(baseResult.getErrMsg());
                            mRootView.killMyself();
                        } else {
                            mRootView.showMessage(AppConstant.GET_INFO_ERROR);
                            mRootView.killMyself();
                        }
                    }

                    /**
                     * 處理單位問題
                     * 未收過貨的單：顯示第一個副單位，用戶可切換副單位和主單位。副單位是 foodUnits集合
                     * 收過貨的單：不可操作單位，顯示入庫單位 purchaseWarehousingUnits
                     * showUnit 用於顯示的單位，進行收貨操作時傳遞給後台
                     */
                    private void transformationWareHouse(PurchaseOrderDetailsBean orderDetailsBean) throws NullPointerException {
                        List<PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean> purchaseOrderDetailsInfoVos = orderDetailsBean.getPurchaseOrderDetailsInfoVOS();
                        if (purchaseOrderDetailsInfoVos != null) {
                            //遍歷訂單商品
                            for (PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean purchaseOrderDetailsInfoVO : purchaseOrderDetailsInfoVos) {
                                purchaseOrderDetailsInfoVO.setOrderNo(orderDetailsBean.getOrderNo());
                                //單位
                                List<DeputyUnitBean> deputyUnitBeans = purchaseOrderDetailsInfoVO.getFoodUnits();
                                //計算出當前食材總金額
                                double warehousingAmount = MoneyUtil.priceCalculation(purchaseOrderDetailsInfoVO.getFoodPrice(), purchaseOrderDetailsInfoVO.getFoodQuantity());
                                //待收貨的
                                if (purchaseOrderDetailsInfoVO.getStatus() == PurchaseOrderDetailsBean.WAIT_RECEIVED) {
                                    //副單位為空，設置顯示主單位
                                    if (deputyUnitBeans == null) {
                                        deputyUnitBeans = new ArrayList<>();
                                    }
                                    if (TextUtil.isEmptyOrNullOrUndefined(purchaseOrderDetailsInfoVO.getBasicUnitName())) {
                                        throw new NullPointerException("未獲取到食品主單位，請檢查後重試");
                                    }
                                    //添加主單位到副單位列表，用戶可手動切換
                                    DeputyUnitBean deputyUnitBean = new DeputyUnitBean(purchaseOrderDetailsInfoVO.getBasicUnitName(), purchaseOrderDetailsInfoVO.getFoodQuantity());
                                    deputyUnitBeans.add(0, deputyUnitBean);
                                    //设置显示的单位，如果大於1，則顯示第一個，因為第一個是副單位，默認顯示副單位
                                    if (deputyUnitBeans.size() > 1) {
                                        purchaseOrderDetailsInfoVO.setShowUnit(deputyUnitBeans.get(1));
                                    } else {
                                        //只有一個單位，顯示主單位
                                        purchaseOrderDetailsInfoVO.setShowUnit(deputyUnitBeans.get(0));
                                    }
                                    //設置默認顯示數量和價格
                                    purchaseOrderDetailsInfoVO.getShowUnit().setDeputyValue(purchaseOrderDetailsInfoVO.getFoodQuantity());
                                    purchaseOrderDetailsInfoVO.getShowUnit().setUnitPrice(purchaseOrderDetailsInfoVO.getFoodPrice());
                                    //設置單位集合
                                    purchaseOrderDetailsInfoVO.setFoodUnits(deputyUnitBeans);
                                } else {
                                    //如果不是待收貨的商品：部分收貨和已完成，都不可再更改收貨單位，只顯示一個固定的單位
                                    //拿到入庫的那個單位
                                    WareHousingUnitBean wareHousingUnitBean = purchaseOrderDetailsInfoVO.getPurchaseWarehousingUnit();
                                    if (wareHousingUnitBean != null) {
                                        //只有一個入庫單位
                                        DeputyUnitBean warehouseUnit = DeputyUnitBean.getDeputyUnitByWareHousingUnit(wareHousingUnitBean);
                                        if (purchaseOrderDetailsInfoVO.getStatus() == PurchaseOrderDetailsBean.COMPLETE_RECEIVED) {
                                            //已經全部收貨，顯示已入庫數量
                                            warehouseUnit.setDeputyValue(purchaseOrderDetailsInfoVO.getWarehousingQuantity());
                                        } else {
                                            //已經入庫一次的食材，數量顯示為食材總數-已入庫數量
                                            warehouseUnit.setDeputyValue(Math.max(MoneyUtil.sub(purchaseOrderDetailsInfoVO.getFoodQuantity(), purchaseOrderDetailsInfoVO.getWarehousingQuantity()), 0));
                                        }
                                        purchaseOrderDetailsInfoVO.setShowUnit(warehouseUnit);
                                        purchaseOrderDetailsInfoVO.setFoodUnits(Collections.singletonList(warehouseUnit));
                                    } else {
                                        wareHousingUnitBean = new WareHousingUnitBean(purchaseOrderDetailsInfoVO.getId(), purchaseOrderDetailsInfoVO.getBasicUnitName(), purchaseOrderDetailsInfoVO.getWarehousingQuantity(), 0);
                                        purchaseOrderDetailsInfoVO.setPurchaseWarehousingUnit(wareHousingUnitBean);
                                        purchaseOrderDetailsInfoVO.setShowUnit(new DeputyUnitBean(purchaseOrderDetailsInfoVO.getBasicUnitName(), purchaseOrderDetailsInfoVO.getWarehousingQuantity()));
                                        mRootView.showMessage("沒有入庫單位,請聯繫開發人員");
                                        mRootView.killMyself();
                                    }
                                }
                                purchaseOrderDetailsInfoVO.setWarehousingAmount(warehousingAmount);
                                purchaseOrderDetailsInfoVO.setTotalAmount(MoneyUtil.priceCalculation(purchaseOrderDetailsInfoVO.getFoodPrice(), purchaseOrderDetailsInfoVO.getFoodQuantity()));
                            }
                        }
                    }

                    @Override
                    public void onError(Throwable t) {
                        super.onError(t);
                        mRootView.showMessage(AppConstant.GET_INFO_ERROR);
                        mRootView.killMyself();
                    }
                });
    }

    /**
     * 加載圖片
     *
     * @param images
     */
    public void loadImages(String images) {
        if (TextUtil.isNotEmptyOrNullOrUndefined(images)) {
            String[] split = images.split(COMMA_SEPARATED);
            if (imgs == null) {
                imgs = new ArrayList<>();
            }
            for (String s : split) {
                if (TextUtil.isNotEmptyOrNullOrUndefined(s)) {
                    mRootView.uploadPicSuccess(s);
                    imgs.add(s);
                }
            }
        }
    }

    /**
     * 入庫
     *
     * @param state                            入庫狀態
     * @param remark                           備註
     * @param totalAmount                      總金額
     * @param purchaseOrderDetailsInfoVosBeans 選中的食品
     */
    public void addWarehouse(int state, String remark, double totalAmount, List<PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean> purchaseOrderDetailsInfoVosBeans) {
        OrderWareHouseBean wareHouseBean = new OrderWareHouseBean();
        List<OrderWareHouseBean.PurchaseWarehousingOrderDetail> purchaseWarehousingOrderDetails = transformWareHoses(wareHouseBean, purchaseOrderDetailsInfoVosBeans);
        wareHouseBean = getWareHouseBean(state, remark, totalAmount, wareHouseBean, purchaseWarehousingOrderDetails);
        if (wareHouseBean == null) {
            mRootView.killMyself();
            return;
        }
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/json"), GsonUtils.GsonString(wareHouseBean));
        mModel.addWarehouse(requestBody)
                .subscribeOn(Schedulers.io())
                .doOnSubscribe(disposable -> mRootView.showLoading(AppConstant.SAVE_LOADING))
                .subscribeOn(AndroidSchedulers.mainThread())
                .observeOn(AndroidSchedulers.mainThread())
                .doAfterTerminate(() -> mRootView.hideLoading())
                .compose(RxLifecycleUtils.bindToLifecycle(mRootView))
                .subscribe(new ErrorHandleSubscriber<BaseResult>(mErrorHandler) {
                    @Override
                    public void onNext(BaseResult baseResult) {
                        if (baseResult.isSuccess()) {
                            mRootView.showMessage("收貨成功");
                            mRootView.killMyself();
                        } else if (baseResult.getData() != null) {
                            NoIsFoodSnTipBean noIsFoodSnTipBean = GsonUtils.GsonToBean(baseResult.getData(), NoIsFoodSnTipBean.class);
                            if (noIsFoodSnTipBean != null) {
                                mRootView.showMessage("食材SN編碼【" + noIsFoodSnTipBean.getList() + "】已存在，請刪除後重試!");
                            } else {
                                mRootView.showMessage(baseResult.getErrMsg());
                            }
                        } else if (TextUtil.isNotEmptyOrNullOrUndefined(baseResult.getErrMsg())) {
                            mRootView.showMessage(baseResult.getErrMsg());
                        } else {
                            mRootView.showMessage(AppConstant.SAVE_FAIL);
                        }
                    }
                });
    }

    /**
     * 再來一單
     */
    public void buyAgainByFoodIds(List<PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean> data) {
        if (data == null || data.size() <= 0) {
            return;
        }
        Map<String, Object> map = new HashMap<>(3);
        AppConstant.addRestaurantId(map);
        AppConstant.addBrandId(map);
        StringBuilder stringBuilder = new StringBuilder();
        for (PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean item : data) {
            stringBuilder.append(item.getPurchaseFoodId()).append(AppConstant.COMMA_SEPARATED);
        }
        map.put("ids", stringBuilder.substring(0, stringBuilder.length() - 1));
        mModel.buyAgainByFoodIds(map)
                .subscribeOn(Schedulers.io())
                .doOnSubscribe(disposable -> mRootView.showLoading(AppConstant.GET_INFO_LOADING))
                .subscribeOn(AndroidSchedulers.mainThread())
                .doAfterTerminate(() -> mRootView.hideLoading())
                .observeOn(AndroidSchedulers.mainThread())
                .compose(RxLifecycleUtils.bindToLifecycle(mRootView))
                .subscribe(new ErrorHandleSubscriber<BaseResult>(mErrorHandler) {
                    @Override
                    public void onNext(BaseResult baseResult) {
                        if (baseResult.isSuccess()) {
                            List<PurchaseFoodBean> purchaseFoodBeans = GsonUtils.jsonToList(baseResult.getData(), PurchaseFoodBean.class);
                            for (PurchaseFoodBean purchaseFoodBean : purchaseFoodBeans) {
                                for (PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean item : data) {
                                    if (purchaseFoodBean.getId() == item.getPurchaseFoodId()) {
                                        purchaseFoodBean.setFoodQuantity(item.getFoodQuantity());
                                        break;
                                    }
                                }
                            }
                            SupplyShoppingCart.getInstance().clear();
                            SupplyShoppingCart.getInstance().addAllFood(purchaseFoodBeans);
                            mRootView.buyAgain();
                        } else {
                            mRootView.showMessage(AppConstant.GET_INFO_ERROR);
                        }
                    }
                });
    }


    /**
     * 生成入庫對象
     */
    private OrderWareHouseBean getWareHouseBean(int state, String remark, double totalAmount, OrderWareHouseBean orderWareHouseBean, List<OrderWareHouseBean.PurchaseWarehousingOrderDetail> purchaseWarehousingOrderDetails) {
        if (CollectionUtils.isNullOrEmpty(purchaseWarehousingOrderDetails)) {
            return null;
        }

        orderWareHouseBean.setStatus(state);
        orderWareHouseBean.setOrderId(orderId);

        OrderWareHouseBean.PurchaseWarehousingOrder purchaseWarehousingOrder = new OrderWareHouseBean.PurchaseWarehousingOrder();
        if (imgs != null) {
            StringBuilder files = new StringBuilder();
            for (Object img : imgs) {
                files.append(img).append(COMMA_SEPARATED);
            }
            String substring = files.substring(0, files.length());
            purchaseWarehousingOrder.setFiles(substring);
        }
        purchaseWarehousingOrder.setRestaurantId(AppConstant.getRestaurantId());
        purchaseWarehousingOrder.setBrandId(AppConstant.getBrandId());
        purchaseWarehousingOrder.setRemarks(remark);
        purchaseWarehousingOrder.setSupplierId(supplierId);
        purchaseWarehousingOrder.setTotalPrice(totalAmount);
        orderWareHouseBean.setPurchaseWarehousingOrder(purchaseWarehousingOrder);
        orderWareHouseBean.setPurchaseWarehousingOrderDetailsTOS(purchaseWarehousingOrderDetails);

        if (purchaseWarehousingOrderDetails.size() <= 0) {
            mRootView.showMessage("請選擇需要操作的食品");
            return null;
        }
        return orderWareHouseBean;
    }

    /**
     * 將訂單轉換為入庫單
     */
    private List<OrderWareHouseBean.PurchaseWarehousingOrderDetail> transformWareHoses(OrderWareHouseBean wareHouseBean, List<PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean> purchaseOrderDetailsInfoVosBeans) {
        if (purchaseOrderDetailsInfoVosBeans == null || purchaseOrderDetailsInfoVosBeans.size() <= 0) {
            mRootView.showMessage(AppConstant.NULL_INFO_ERROR);
            return null;
        }
        List<OrderWareHouseBean.PurchaseWarehousingOrderDetail> purchaseWarehousingOrderDetails = new ArrayList<>(purchaseOrderDetailsInfoVosBeans.size());
        List<OrderWareHouseBean.PurchaseConsumeVerifyEncodeSn> purchaseConsumeVerifyEncodeSns = new ArrayList<>(purchaseOrderDetailsInfoVosBeans.size());
        for (PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean purchaseOrderDetailsInfoVosBean : purchaseOrderDetailsInfoVosBeans) {
            if (purchaseOrderDetailsInfoVosBean.getStatus() != PurchaseOrderDetailsBean.COMPLETE_RECEIVED && purchaseOrderDetailsInfoVosBean.isChecked()) {
                //新的入庫類
                OrderWareHouseBean.PurchaseWarehousingOrderDetail purchaseWarehousingOrderDetail = new OrderWareHouseBean.PurchaseWarehousingOrderDetail();
                purchaseWarehousingOrderDetail.setOrderDetailsId(purchaseOrderDetailsInfoVosBean.getId());
                purchaseWarehousingOrderDetail.setPurchaseFoodId(purchaseOrderDetailsInfoVosBean.getPurchaseFoodId());
                purchaseWarehousingOrderDetail.setFoodQuantity(purchaseOrderDetailsInfoVosBean.getFoodQuantity());
                purchaseWarehousingOrderDetail.setFoodPrice(purchaseOrderDetailsInfoVosBean.getFoodPrice());
                purchaseWarehousingOrderDetail.setSupplierId(supplierId);
                purchaseWarehousingOrderDetail.setTotalPrice(MoneyUtil.priceCalculation(purchaseOrderDetailsInfoVosBean.getFoodPrice(), purchaseOrderDetailsInfoVosBean.getFoodQuantity()));
                DeputyUnitBean deputyUnitBean = purchaseOrderDetailsInfoVosBean.getShowUnit();
                WareHousingUnitBean wareHousingUnitBean = DeputyUnitBean.getWareHousingUnitByDeputyUnit(deputyUnitBean);
//                if(wareHousingUnitBean.getUnitValue()<=0){
//                    //如果入庫數量小於等於0，就不管這個食品
//                    continue;
//                }
                if (purchaseOrderDetailsInfoVosBean.getStatus() != PurchaseOrderDetailsBean.WAIT_RECEIVED && deputyUnitBean.getId() != null) {
                    //不是待收貨狀態才設置id，因為如果已經收過貨才有id
                    wareHousingUnitBean.setId(deputyUnitBean.getId());
                }
                wareHousingUnitBean.setBrandId(AppConstant.getBrandId());
                wareHousingUnitBean.setRestaurantId(AppConstant.getRestaurantId());
                purchaseWarehousingOrderDetail.setPurchaseWarehousingUnit(wareHousingUnitBean);
                //sn碼
                List<PurchaseFoodEncodeSn> snList = null;
                List<String> snCodes;
                if (purchaseOrderDetailsInfoVosBean.getPurchaseFoodEncodeSns() != null) {
                    snList = new ArrayList<>(purchaseOrderDetailsInfoVosBean.getPurchaseFoodEncodeSns().size());
                    snCodes = new ArrayList<>(purchaseOrderDetailsInfoVosBean.getPurchaseFoodEncodeSns().size());
                    //只遍歷出新增的
                    for (PurchaseFoodEncodeSn purchaseFoodEncodeSn : purchaseOrderDetailsInfoVosBean.getPurchaseFoodEncodeSns()) {
                        if (purchaseFoodEncodeSn.newAdd) {
                            snList.add(purchaseFoodEncodeSn);
                            snCodes.add(purchaseFoodEncodeSn.getEncodeSnNo());
                        }
                    }
                    purchaseConsumeVerifyEncodeSns.add(new OrderWareHouseBean.PurchaseConsumeVerifyEncodeSn(purchaseOrderDetailsInfoVosBean.getFoodNo(), snCodes));
                }
                purchaseWarehousingOrderDetail.setPurchaseFoodEncodeSn(snList);
                purchaseWarehousingOrderDetails.add(purchaseWarehousingOrderDetail);
            }
        }
        wareHouseBean.setPurchaseConsumeVerifyEncodeSn(purchaseConsumeVerifyEncodeSns);
        return purchaseWarehousingOrderDetails;
    }

    /**
     * 刪除訂單
     */
    public void deleteOrder() {
        mModel.deleteOrder(orderId)
                .subscribeOn(Schedulers.io())
                .doOnSubscribe(disposable -> mRootView.showLoading(AppConstant.DELETE_LOADING))
                .subscribeOn(AndroidSchedulers.mainThread())
                .observeOn(AndroidSchedulers.mainThread())
                .doAfterTerminate(() -> mRootView.hideLoading())
                .compose(RxLifecycleUtils.bindToLifecycle(mRootView))
                .subscribe(new ErrorHandleSubscriber<BaseResult>(mErrorHandler) {
                    @Override
                    public void onNext(@NonNull BaseResult info) {
                        if (info.isSuccess()) {
                            mRootView.showMessage(mApplication.getString(R.string.str_order_del_contact_supplier));
                            mRootView.killMyself();
                        }
                    }
                });
    }

    /**
     * 是否全是待收货食材
     *
     * @param data 食材信息
     * @return
     */
    public boolean allWaitReceived(List<PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean> data) {
        if (data != null) {
            for (PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean datum : data) {
                if (datum.getStatus() != PurchaseOrderDetailsBean.WAIT_RECEIVED) {
                    return false;
                }
            }
            return true;
        }
        return true;
    }

    /**
     * 是否有sn食材
     *
     * @param purchaseOrderDetailsInfoVOS 食材信息
     * @return
     */
    public boolean hasSnFood(List<PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean> purchaseOrderDetailsInfoVOS) {
        if (purchaseOrderDetailsInfoVOS != null) {
            for (PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean datum : purchaseOrderDetailsInfoVOS) {
                if (datum.getFoodMarkSn() == PurchaseFoodBean.HAS_SN) {
                    return true;
                }
            }
            return false;
        }
        return false;
    }

    /**
     * 刪除Sn碼
     */
    public void deleteSn(String enCodeOrderNo, String encodeFoodNo, String encodeSnNo) {
        Map<String, Object> map = new HashMap<>(3);
        map.put("encodeOrderNo", enCodeOrderNo);
        map.put("encodeFoodNo", encodeFoodNo);
        map.put("encodeSnNo", encodeSnNo);
        mModel.deleteSnCode(map)
                .subscribeOn(Schedulers.io())
                .doOnSubscribe(disposable -> mRootView.showLoading(AppConstant.DELETE_LOADING))
                .subscribeOn(AndroidSchedulers.mainThread())
                .observeOn(AndroidSchedulers.mainThread())
                .doAfterTerminate(() -> mRootView.hideLoading())
                .compose(RxLifecycleUtils.bindToLifecycle(mRootView))
                .subscribe(new ErrorHandleSubscriber<BaseResult>(mErrorHandler) {
                    @Override
                    public void onNext(@NonNull BaseResult info) {
                        if (info.isSuccess()) {
                            mRootView.showMessage(AppConstant.DELETE_SUCCESS);

                        }
                    }
                });
    }

    /**
     * 查詢商品sn碼
     */
    public void querySnCodes(int position, PurchaseOrderDetailsBean.PurchaseOrderDetailsInfoVosBean infoVosBean, boolean singerFood) {
        mModel.querySnCodes(infoVosBean.getFoodNo(), infoVosBean.getOrderNo())
                .subscribeOn(Schedulers.io())
                .doOnSubscribe(disposable -> mRootView.showLoading(AppConstant.QUERY_LOADING))
                .subscribeOn(AndroidSchedulers.mainThread())
                .observeOn(AndroidSchedulers.mainThread())
                .doAfterTerminate(() -> mRootView.hideLoading())
                .compose(RxLifecycleUtils.bindToLifecycle(mRootView))
                .subscribe(new ErrorHandleSubscriber<BaseResult>(mErrorHandler) {
                    @Override
                    public void onNext(@NonNull BaseResult info) {
                        if (info.isSuccess()) {
                            List<PurchaseFoodEncodeSn> purchaseFoodEncodeSns = GsonUtils.jsonToList(info.getData(), PurchaseFoodEncodeSn.class);
                            if (purchaseFoodEncodeSns != null) {
                                if (infoVosBean.getPurchaseFoodEncodeSns() == null) {
                                    infoVosBean.setPurchaseFoodEncodeSns(purchaseFoodEncodeSns);
                                } else {
                                    for (PurchaseFoodEncodeSn purchaseFoodEncodeSn : purchaseFoodEncodeSns) {
                                        if (!infoVosBean.getPurchaseFoodEncodeSns().contains(purchaseFoodEncodeSn)) {
                                            infoVosBean.getPurchaseFoodEncodeSns().add(purchaseFoodEncodeSn);
                                        }
                                    }
                                }
                            }
                            mRootView.querySnCodeSuccess(position, infoVosBean, singerFood);
                        } else if (!singerFood) {
                            mRootView.nextFood();
                        }
                    }

                    @Override
                    public void onError(Throwable t) {
                        super.onError(t);
                        if (!singerFood) {
                            mRootView.nextFood();
                        }
                    }
                });
    }

    /**
     * 上傳圖片
     *
     * @param filePath 圖片路徑
     */
    public void uploadPic(String filePath) {
        File file = new File(BitmapUtil.compressImage(filePath));
        if (file.length() <= 0) {
            mRootView.showMessage("圖片已損壞，請重新選擇圖片");
            return;
        }
        List<MultipartBody.Part> multipartBodies = new ArrayList<>();
        multipartBodies.add(MultipartBody.Part.createFormData("files", file.getName(), RequestBody.create(MediaType.parse("image/png"), file)));
        Map<String, Object> map = new HashMap<>(1);
        map.put("type", PurchaseConstant.UPLOAD_PIC_TYPE_ORDER);
        mModel.upLoadPic(multipartBodies, map)
                .subscribeOn(Schedulers.io())
                .doOnSubscribe(disposable -> mRootView.showLoading(AppConstant.IMG_UPLOAD_LOADING))
                .subscribeOn(AndroidSchedulers.mainThread())
                .observeOn(AndroidSchedulers.mainThread())
                .doAfterTerminate(() -> mRootView.hideLoading())
                .compose(RxLifecycleUtils.bindToLifecycle(mRootView))
                .subscribe(new ErrorHandleSubscriber<BaseResult>(mErrorHandler) {
                    @Override
                    public void onNext(BaseResult baseResult) {
                        if (baseResult.isSuccess() && baseResult.getData() != null) {
                            List<String> imgUrl = JsonUtils.parseArray(baseResult.getData(), String.class);
                            if (imgUrl != null && imgUrl.size() > 0) {
                                String picUrl = imgUrl.get(0);
                                if (imgs == null) {
                                    imgs = new ArrayList<>();
                                }
                                imgs.add(picUrl);
                                mRootView.uploadPicSuccess(picUrl);
                            }
                        }
                        boolean delete = file.delete();
                    }
                });
    }

    public List<Object> getImgs() {
        return imgs;
    }

    public void deletePic(int position) {
        if (imgs != null && position < imgs.size()) {
            imgs.remove(position);
        }
    }

    public Object getPic(int position) {
        if (imgs != null && position < imgs.size()) {
            return imgs.get(position);
        }
        return null;
    }

    public int getNewSnCodeSize(List<PurchaseFoodEncodeSn> purchaseFoodEncodeSns) {
        if (purchaseFoodEncodeSns == null) {
            return 0;
        }
        int size = 0;
        for (PurchaseFoodEncodeSn purchaseFoodEncodeSn : purchaseFoodEncodeSns) {
            if (purchaseFoodEncodeSn.newAdd) {
                size++;
            }
        }
        return size;
    }

    public int getPicSize() {
        return imgs == null ? 0 : imgs.size();
    }
}
