package com.gingersoft.gsa.cloud.common.utils;

import android.annotation.SuppressLint;
import android.content.ContentUris;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.provider.DocumentsContract;
import android.provider.MediaStore;
import android.text.TextUtils;

import com.elvishew.xlog.XLog;
import com.gingersoft.gsa.cloud.common.utils.encryption.BASE64Encoder;
import com.gingersoft.gsa.cloud.common.utils.view.BitmapUtil;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;

/**
 * 作者：ELEGANT_BIN
 * 版本：1.6.0
 * 创建日期：2018/11/16
 * 修订历史：2018/11/16
 * 描述：7.0 兼容
 * <files-path name="name" path="path" /> 对应getFilesDir()。
 * <cache-path name="name" path="path" /> 对应getCacheDir()。
 * <external-path name="name" path="path" /> 对应Environment.getExternalStorageDirectory()。
 * <external-files-path name="name" path="path" /> 对应getExternalFilesDir()。
 * <external-cache-path name="name" path="path" /> 对应getExternalCacheDir()。
 */
public class FileUtils {

    //一级目录
    public static String FirstFolder = "GSACloud";
    //文件文件夹
    public static String file = "file";
    //图片文件夹
    public static String Image = "image";
    //日志存储
    public static String Log = "xLog";
    //操作日志
    public static String ActionLog = "actionLog";
    //错误日志存储
    public static String ErrorLog = "errorLog";
    //安装包目录放在系统目录
    public static String APK = "Download";
    /*ALBUM_PATH取得机器的SD卡位置，File.separator为分隔符“/”*/
    public final static String ALBUM_PATH = Environment.getExternalStorageDirectory() + File.separator + FirstFolder + File.separator;
    public final static String IMAGE_PATH = ALBUM_PATH + Image + File.separator;
    public final static String File_PATH = ALBUM_PATH + file + File.separator;
    public final static String LOG_PATH = ALBUM_PATH + Log + File.separator;
    public final static String ACTIONLOG_PATH = LOG_PATH + ActionLog + File.separator;
    public final static String ERRORLOG_PATH = LOG_PATH + ErrorLog + File.separator;
    public final static String APK_PATH = Environment.getExternalStorageDirectory() + File.separator + APK + File.separator;

    /**
     * 打印前圖片存儲路徑
     */
    public final static class PrinterImagePath {
        //Prj圖片文件最大文件大小不能超過100M
        public static final int PRINTER_MAX_FILE_SIZE = 100 * 1024 * 1024;
        //Prj最少保留文件大小
        public static final int PRINTER_KEEP_FILE_SIZE = 20 * 1024 * 1024;

        public static String PrinterImagesRootPath = "PrinterImages";
        public static String KitchenImage = "KitchenPrj";
        public static String OrderImage = "OrderImage";
        public static String BillImage = "BillImage";
        public static String ServingPaperImage = "ServingPaperImage";
        public final static String ROOT_PATH = ALBUM_PATH + PrinterImagesRootPath + File.separator;
        public final static String KITCHEN_PRJ_IMAGE_PATH = ROOT_PATH + KitchenImage;
        public final static String ORDER_IMAGE_PATH = ROOT_PATH + OrderImage;
        public final static String BILL_IMAGE_PATH = ROOT_PATH + BillImage;
        public final static String SERVING_PAPER_IMAGE_PATH = ROOT_PATH + ServingPaperImage;
    }

    public final static String FILE_EXTENSION_SEPARATOR = ".";

    //操作日志單最大文件大小
    public final static int ACTION_MAX_SIZE = 1024 * 1024 * 5;
    //操作日志文件按照時間刪除
    public final static int ACTION_MAX_FILE_TIME = 1000 * 60 * 60 * 12;

    private FileUtils() {
        throw new AssertionError();
    }

    public static StringBuilder readFile(String filePath, String charsetName) {
        File file = new File(filePath);
        StringBuilder fileContent = new StringBuilder("");
        if (file == null || !file.isFile()) {
            return null;
        }
        BufferedReader reader = null;
        try {
            InputStreamReader is = new InputStreamReader(new FileInputStream(
                    file), charsetName);
            reader = new BufferedReader(is);
            String line = null;
            while ((line = reader.readLine()) != null) {
                if (!fileContent.toString().equals("")) {
                    fileContent.append("\r\n");
                }
                fileContent.append(line);
            }
            reader.close();
            return fileContent;
        } catch (IOException e) {
            throw new RuntimeException("IOException occurred. ", e);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    throw new RuntimeException("IOException occurred. ", e);
                }
            }
        }
    }

    /**
     * 写文件
     *
     * @param
     * @return boolean 返回类型
     */
    public static boolean writeFile(String filePath, String content,
                                    boolean append) {
        if (TextUtils.isEmpty(content)) {
            return false;
        }

        FileWriter fileWriter = null;
        try {
            makeDirs(filePath);
            fileWriter = new FileWriter(filePath, append);
            fileWriter.write(content);
            fileWriter.close();
            return true;
        } catch (IOException e) {
            throw new RuntimeException("IOException occurred. ", e);
        } finally {
            if (fileWriter != null) {
                try {
                    fileWriter.close();
                } catch (IOException e) {
                    throw new RuntimeException("IOException occurred. ", e);
                }
            }
        }
    }

    /**
     * 创建文件的路径
     *
     * @param
     * @return boolean 返回类型
     */
    public static boolean makeDirs(String filePath) {
        String folderName = getFolderName(filePath);
        if (TextUtils.isEmpty(folderName)) {
            return false;
        }

        File folder = new File(folderName);
        return (folder.exists() && folder.isDirectory()) ? true : folder.mkdirs();
    }

    public static String getFolderName(String filePath) {

        if (TextUtils.isEmpty(filePath)) {
            return filePath;
        }

        int filePosi = filePath.lastIndexOf(File.separator);
        return (filePosi == -1) ? "" : filePath.substring(0, filePosi);
    }

    /**
     * 写文本文件 在Android系统中，文件保存在 /data/data/PACKAGE_NAME/files 目录下
     *
     * @param context
     */
    public static void write(Context context, String fileName, String content) {
        if (content == null) {
            content = "";
        }
        FileOutputStream fos = null;
        try {
            fos = context.openFileOutput(fileName, Context.MODE_PRIVATE);
            fos.write(content.getBytes());
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (fos != null) {
                closeQuietly(fos);
            }
        }
    }

    /**
     * 读取文本文件 文件保存在 /data/data/PACKAGE_NAME/files
     *
     * @param context
     * @param fileName
     * @return
     */
    public static String read(Context context, String fileName) {
        try {
            FileInputStream in = context.openFileInput(fileName);
            return readInStream(in);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

    public static String readInStream(FileInputStream inStream) {
        try {
            ByteArrayOutputStream outStream = new ByteArrayOutputStream();
            byte[] buffer = new byte[512];
            int length = -1;
            while ((length = inStream.read(buffer)) != -1) {
                outStream.write(buffer, 0, length);
            }
            outStream.close();
            inStream.close();
            return outStream.toString();
        } catch (IOException e) {
            XLog.tag("FileTest").d(e.getMessage());
        }
        return null;
    }

    /**
     * 创建文件
     *
     * @param folderPath
     * @param fileName
     * @return
     */
    public static File createFile(String folderPath, String fileName) {
        File destDir = new File(folderPath);
        if (!destDir.exists()) {
            destDir.mkdirs();
        }
        return new File(folderPath, fileName + fileName);
    }

    /*
     * 在SD卡上创建目录
     */
    public static File creatSDDir(String dirName) {
        File dir = new File(dirName);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        return dir;
    }

    /**
     * 获取文件大小
     *
     * @param filePath
     * @return
     */
    public static long getFileSize(String filePath) {
        long size = 0;

        File file = new File(filePath);
        if (file != null && file.exists()) {
            size = file.length();
        }
        return size;
    }

    /*
     * Java文件操作 获取文件扩展名
     * */
    public static String getExtensionName(String filename) {
        if ((filename != null) && (filename.length() > 0)) {
            int dot = filename.lastIndexOf('.');
            if ((dot > -1) && (dot < (filename.length() - 1))) {
                return filename.substring(dot + 1);
            }
        }
        return filename;
    }

    /*
     * Java文件操作 获取不带扩展名的文件名
     * */
    public static String getFileNameNoEx(String filename) {
        if ((filename != null) && (filename.length() > 0)) {
            int dot = filename.lastIndexOf('.');
            if ((dot > -1) && (dot < (filename.length()))) {
                return filename.substring(0, dot);
            }
        }
        return filename;
    }


    public static void closeQuietly(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            } catch (RuntimeException rethrown) {
                throw rethrown;
            } catch (Exception ignored) {
            }
        }
    }

    /**
     * 根据URI获取文件真实路径（兼容多张机型）
     *
     * @param context
     * @param uri
     * @return
     */
    public static String getFilePathByUri(Context context, Uri uri) {
        if ("content".equalsIgnoreCase(uri.getScheme())) {

            int sdkVersion = Build.VERSION.SDK_INT;
            if (sdkVersion >= 19) { // api >= 19
                return getRealPathFromUriAboveApi19(context, uri);
            } else { // api < 19
                return getRealPathFromUriBelowAPI19(context, uri);
            }
        } else if ("file".equalsIgnoreCase(uri.getScheme())) {
            return uri.getPath();
        }
        return null;
    }

    /**
     * 适配api19及以上,根据uri获取图片的绝对路径
     *
     * @param context 上下文对象
     * @param uri     图片的Uri
     * @return 如果Uri对应的图片存在, 那么返回该图片的绝对路径, 否则返回null
     */
    @SuppressLint("NewApi")
    private static String getRealPathFromUriAboveApi19(Context context, Uri uri) {
        String filePath = null;
        if (DocumentsContract.isDocumentUri(context, uri)) {
            // 如果是document类型的 uri, 则通过document id来进行处理
            String documentId = DocumentsContract.getDocumentId(uri);
            if (isMediaDocument(uri)) { // MediaProvider
                // 使用':'分割
                String type = documentId.split(":")[0];
                String id = documentId.split(":")[1];

                String selection = MediaStore.Images.Media._ID + "=?";
                String[] selectionArgs = {id};

                //
                Uri contentUri = null;
                if ("image".equals(type)) {
                    contentUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI;
                } else if ("video".equals(type)) {
                    contentUri = MediaStore.Video.Media.EXTERNAL_CONTENT_URI;
                } else if ("audio".equals(type)) {
                    contentUri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI;
                }

                filePath = getDataColumn(context, contentUri, selection, selectionArgs);
            } else if (isDownloadsDocument(uri)) { // DownloadsProvider
                Uri contentUri = ContentUris.withAppendedId(Uri.parse("content://downloads/public_downloads"), Long.valueOf(documentId));
                filePath = getDataColumn(context, contentUri, null, null);
            } else if (isExternalStorageDocument(uri)) {
                // ExternalStorageProvider
                final String docId = DocumentsContract.getDocumentId(uri);
                final String[] split = docId.split(":");
                final String type = split[0];
                if ("primary".equalsIgnoreCase(type)) {
                    filePath = Environment.getExternalStorageDirectory() + "/" + split[1];
                }
            } else {
                //Log.e("路径错误");
            }
        } else if ("content".equalsIgnoreCase(uri.getScheme())) {
            // 如果是 content 类型的 Uri
            filePath = getDataColumn(context, uri, null, null);
        } else if ("file".equals(uri.getScheme())) {
            // 如果是 file 类型的 Uri,直接获取图片对应的路径
            filePath = uri.getPath();
        }
        return filePath;
    }

    /**
     * 适配api19以下(不包括api19),根据uri获取图片的绝对路径
     *
     * @param context 上下文对象
     * @param uri     图片的Uri
     * @return 如果Uri对应的图片存在, 那么返回该图片的绝对路径, 否则返回null
     */
    private static String getRealPathFromUriBelowAPI19(Context context, Uri uri) {
        return getDataColumn(context, uri, null, null);
    }

    /**
     * 获取数据库表中的 _data 列，即返回Uri对应的文件路径
     *
     * @return
     */
    private static String getDataColumn(Context context, Uri uri, String selection, String[] selectionArgs) {
        String path = null;

        String[] projection = new String[]{MediaStore.Images.Media.DATA};
        Cursor cursor = null;
        try {
            cursor = context.getContentResolver().query(uri, projection, selection, selectionArgs, null);
            if (cursor != null && cursor.moveToFirst()) {
                int columnIndex = cursor.getColumnIndexOrThrow(projection[0]);
                path = cursor.getString(columnIndex);
            }
        } catch (Exception e) {
            if (cursor != null) {
                cursor.close();
            }
        }
        return path;
    }

    /**
     * @param uri the Uri to check
     * @return Whether the Uri authority is MediaProvider
     */
    private static boolean isMediaDocument(Uri uri) {
        return "com.android.providers.media.documents".equals(uri.getAuthority());
    }

    private static boolean isExternalStorageDocument(Uri uri) {
        return "com.android.externalstorage.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri the Uri to check
     * @return Whether the Uri authority is DownloadsProvider
     */
    private static boolean isDownloadsDocument(Uri uri) {
        return "com.android.providers.downloads.documents".equals(uri.getAuthority());
    }

    /**
     * 将图片编码成 Base64字符串
     */
    public static String Encode_To_Base64(String IMAGE_URL) {
        File file = new File(IMAGE_URL);
        FileInputStream inputStream;
        try {
            inputStream = new FileInputStream(IMAGE_URL);
            byte[] buffer = new byte[(int) file.length()];
            inputStream.read(buffer);
            inputStream.close();
            BASE64Encoder encoder = new BASE64Encoder();

            return encoder.encode(buffer);
        } catch (IOException e) {
            e.printStackTrace();
            return "";
        }
    }


    /**
     * 复制文件
     *
     * @param fromfile
     * @param tofile
     */
    public static void copyFile(String fromfile, String tofile) {
        try {
            FileInputStream fosfrom = new FileInputStream(fromfile);
            FileOutputStream fosto = new FileOutputStream(tofile);
            byte[] b = new byte[1024];
            int c;
            while ((c = fosfrom.read(b)) > 0) {
                //将内容写到新文件当中
                fosto.write(b, 0, c);
            }
            fosfrom.close();
            fosto.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}
