package com.gingersoft.gsa.cloud.pay.pos;

import android.app.Activity;
import android.content.Context;

import com.etps.aca.lib.util.CommandUtil;
import com.gingersoft.gsa.cloud.pay.pos.bean.PosMatter;
import com.gingersoft.gsa.cloud.pay.pos.hywebpos.bean.response.pos.HywebPosResponse;
import com.gingersoft.gsa.cloud.pay.pos.hywebpos.constant.PayStatusCode;
import com.gingersoft.gsa.cloud.pay.widget.StatusLoadingDialog;
import com.jess.arms.mvp.IView;


/**
 * 作者：ELEGANT_BIN
 * 版本：1.6.0
 * 创建日期：2019/6/13
 * 修订历史：2019/6/13
 * 描述：
 */
public abstract class IPosAction {

    protected Context mContext;
    protected IView mView;
    protected PosAction mPosAction;
    protected OnPosActionListener mOnPosActionListener;

    protected StatusLoadingDialog mStatusLoadingDialog;

    private final String mLoadingActionText = "中...";
    private final String mLoadingActionComplteText = "成功";

    private final long mCancelDelayMillis = 2000;

    public IPosAction(Context context, IView iView, PosAction action, boolean showLoading) {
        this.mContext = context;
        this.mView = iView;
        this.mPosAction = action;
        if (showLoading) {
            mStatusLoadingDialog = new StatusLoadingDialog(context, action.getActionName() + mLoadingActionText, mCancelDelayMillis);
            mStatusLoadingDialog.show();
        }
    }

    /**
     * 對POS幾發起操作前的業務處理,如將操作記錄添加到後台日誌表，事項表
     */
    protected abstract void onBeforeActionPOS();


    /**
     * 向POS發起操作的方法
     */
    protected void onActionToPOS(String requestJson) {
        CommandUtil.sendRequest(mContext, requestJson);
    }

    /**
     * 對POS幾發起操作前的業務處理异常 如记录事项调用出错
     *
     * @param e
     */
    public void onBeforeActionPosError(Exception e) {
        if (mStatusLoadingDialog != null) {
            mStatusLoadingDialog.setStatusText(mPosAction.getActionName() + e.getMessage());
            mStatusLoadingDialog.cancel();
        }
    }

    /**
     * POS回調成功的方法
     *
     * @param posResponse
     */
    public void onPosSuccess(Context context, HywebPosResponse posResponse, Activity originalActivity) {
        if (mStatusLoadingDialog != null) {
            mStatusLoadingDialog.setStatusText(mPosAction.getActionName() + mLoadingActionComplteText);
            mStatusLoadingDialog.loadSuccess();
        }
        if (mOnPosActionListener != null) {
            mOnPosActionListener.onPosSuccess(posResponse);
        }
    }

    /**
     * POS回調為null或狀態異常
     */
    public void onPosException(Exception e) {
        if (mStatusLoadingDialog != null) {
            mStatusLoadingDialog.setStatusText(mPosAction.getCallbackExceptionText() + e.getMessage());
            mStatusLoadingDialog.loadFailure();
        }
        if (mOnPosActionListener != null) {
            mOnPosActionListener.onPosException(e);
        }
    }

    /**
     * POS調起失敗
     */
    public void onPosInvalid(String e) {
        if (mStatusLoadingDialog != null) {
            mStatusLoadingDialog.setStatusText(mPosAction.getActionName() + e);
            mStatusLoadingDialog.loadFailure();
            mStatusLoadingDialog.cancel();
        }
        if (mOnPosActionListener != null) {
            mOnPosActionListener.onPosInvalid(e);
        }
    }

    /**
     * 更新狀態成功
     *
     * @param response
     */
    public void onUpdateStatusSuccess(HywebPosResponse response) {
        if (mStatusLoadingDialog != null) {
            mStatusLoadingDialog.setStatusText(mPosAction.getActionName() + PayStatusCode.getHywebRequestTextByCode(response.getSTATUS()));
            mStatusLoadingDialog.cancel();
        }
        if (mOnPosActionListener != null) {
            mOnPosActionListener.onUpdateStatusSuccess(response);
        }
    }

    /**
     * 更新狀態失败
     *
     * @param e
     */
    public void onUpdateStatusError(Exception e) {
        if (mStatusLoadingDialog != null) {
            mStatusLoadingDialog.setStatusText(mPosAction.getActionName() + e.getMessage());
            mStatusLoadingDialog.cancel();
        }
        if (mOnPosActionListener != null) {
            mOnPosActionListener.onUpdateStatusError(e);
        }
    }

    protected void canCelLoadingDialog() {
        if (mStatusLoadingDialog != null) {
            mStatusLoadingDialog.cancel();
        }
    }

    /**
     * 根據業務回調狀態，返回事項上傳狀態
     *
     * @param status
     * @return
     */
    protected byte getPosMatterStatus(String status) {
        if (PayStatusCode.withSuccess(status)) {
            return PosMatter.status_1;
        } else if (PayStatusCode.withException(status)) {
            return PosMatter.status_0;
        } else {
            return PosMatter.status_2;
        }
    }

}
