/*
 * Copyright 2017 JessYan
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jess.arms.base.delegate;


import android.app.Activity;
import android.os.Bundle;
import android.view.WindowManager;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.FragmentManager;

import com.gyf.immersionbar.BarHide;
import com.gyf.immersionbar.ImmersionBar;
import com.gyf.immersionbar.OnBarListener;
import com.gyf.immersionbar.OnNavigationBarListener;
import com.jess.arms.R;
import com.jess.arms.base.BaseActivity;
import com.jess.arms.base.BaseFragment;
import com.jess.arms.di.component.AppComponent;
import com.jess.arms.integration.ActivityLifecycle;
import com.jess.arms.integration.cache.Cache;
import com.jess.arms.integration.cache.LruCache;

import org.simple.eventbus.EventBus;

/**
 * ================================================
 * 框架要求框架中的每个 {@link Activity} 都需要实现此类,以满足规范
 *
 * @see BaseActivity
 * Created by JessYan on 26/04/2017 21:42
 * <a href="mailto:jess.yan.effort@gmail.com">Contact me</a>
 * <a href="https://github.com/JessYanCoding">Follow me</a>
 * ================================================
 */
public interface IActivity {

    /**
     * 提供在 {@link Activity} 生命周期内的缓存容器, 可向此 {@link Activity} 存取一些必要的数据
     * 此缓存容器和 {@link Activity} 的生命周期绑定, 如果 {@link Activity} 在屏幕旋转或者配置更改的情况下
     * 重新创建, 那此缓存容器中的数据也会被清空, 如果你想避免此种情况请使用 <a href="https://github.com/JessYanCoding/LifecycleModel">LifecycleModel</a>
     *
     * @return like {@link LruCache}
     */
    @NonNull
    Cache<String, Object> provideCache();

    /**
     * 提供 AppComponent (提供所有的单例对象) 给实现类, 进行 Component 依赖
     *
     * @param appComponent
     */
    void setupActivityComponent(@NonNull AppComponent appComponent);

    /**
     * 是否使用 {@link EventBus}
     *
     * @return
     */
    boolean useEventBus();

    /**
     * 初始化 View, 如果 {@link #initView(Bundle)} 返回 0, 框架则不会调用 {@link Activity#setContentView(int)}
     *
     * @param savedInstanceState
     * @return
     */
    int initView(@Nullable Bundle savedInstanceState);

    /**
     * 初始化数据
     *
     * @param savedInstanceState
     */
    void initData(@Nullable Bundle savedInstanceState);

    /**
     * 重写此方法可配置当前页面的沉浸式主题
     * ImmersionBar.with(this)
     *                 .transparentStatusBar()  //透明状态栏，不写默认透明色
     *                 .transparentNavigationBar()  //透明导航栏，不写默认黑色(设置此方法，fullScreen()方法自动为true)
     *                 .transparentBar()             //透明状态栏和导航栏，不写默认状态栏为透明色，导航栏为黑色（设置此方法，fullScreen()方法自动为true）
     *                 .statusBarColor(R.color.colorPrimary)     //状态栏颜色，不写默认透明色
     *                 .navigationBarColor(R.color.colorPrimary) //导航栏颜色，不写默认黑色
     *                 .barColor(R.color.colorPrimary)  //同时自定义状态栏和导航栏颜色，不写默认状态栏为透明色，导航栏为黑色
     *                 .statusBarAlpha(0.3f)  //状态栏透明度，不写默认0.0f
     *                 .navigationBarAlpha(0.4f)  //导航栏透明度，不写默认0.0F
     *                 .barAlpha(0.3f)  //状态栏和导航栏透明度，不写默认0.0f
     *                 .statusBarDarkFont(true)   //状态栏字体是深色，不写默认为亮色
     *                 .navigationBarDarkIcon(true) //导航栏图标是深色，不写默认为亮色
     *                 .autoDarkModeEnable(true) //自动状态栏字体和导航栏图标变色，必须指定状态栏颜色和导航栏颜色才可以自动变色哦
     *                 .autoStatusBarDarkModeEnable(true,0.2f) //自动状态栏字体变色，必须指定状态栏颜色才可以自动变色哦
     *                 .autoNavigationBarDarkModeEnable(true,0.2f) //自动导航栏图标变色，必须指定导航栏颜色才可以自动变色哦
     *                 .flymeOSStatusBarFontColor(R.color.black)  //修改flyme OS状态栏字体颜色
     *                 .fullScreen(true)      //有导航栏的情况下，activity全屏显示，也就是activity最下面被导航栏覆盖，不写默认非全屏
     *                 .hideBar(BarHide.FLAG_HIDE_BAR)  //隐藏状态栏或导航栏或两者，不写默认不隐藏
     *                 .addViewSupportTransformColor(view)  //设置支持view变色，可以添加多个view，不指定颜色，默认和状态栏同色，还有两个重载方法
     *                 .titleBar(view)    //解决状态栏和布局重叠问题，任选其一
     *                 .titleBarMarginTop(view)     //解决状态栏和布局重叠问题，任选其一
     *                 .statusBarView(view)  //解决状态栏和布局重叠问题，任选其一
     *                 .fitsSystemWindows(true)    //解决状态栏和布局重叠问题，任选其一，默认为false，当为true时一定要指定statusBarColor()，不然状态栏为透明色，还有一些重载方法
     *                 .supportActionBar(true) //支持ActionBar使用
     *                 .statusBarColorTransform(R.color.theme_color)  //状态栏变色后的颜色
     *                 .navigationBarColorTransform(R.color.theme_color) //导航栏变色后的颜色
     *                 .barColorTransform(R.color.theme_color)  //状态栏和导航栏变色后的颜色
     *                 .removeSupportView(view)  //移除指定view支持
     *                 .removeSupportAllView() //移除全部view支持
     *                 .navigationBarEnable(true)   //是否可以修改导航栏颜色，默认为true
     *                 .navigationBarWithKitkatEnable(true)  //是否可以修改安卓4.4和emui3.x手机导航栏颜色，默认为true
     *                 .navigationBarWithEMUI3Enable(true) //是否可以修改emui3.x手机导航栏颜色，默认为true
     *                 .keyboardEnable(true)  //解决软键盘与底部输入框冲突问题，默认为false，还有一个重载方法，可以指定软键盘mode
     *                 .keyboardMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_RESIZE)  //单独指定软键盘模式
     *                 .setOnNavigationBarListener(new OnNavigationBarListener() {
     *                     @Override
     *                     public void onNavigationBarChange(boolean show) {
     *
     *                     }
     *                 }) //导航栏显示隐藏监听，目前只支持华为和小米手机
     *                 .setOnBarListener(new OnBarListener() {
     *                     @Override
     *                     public void onBarChange(BarProperties barProperties) {
     *
     *                     }
     *                 }) //第一次调用和横竖屏切换都会触发，可以用来做刘海屏遮挡布局控件的问题
     *                 .addTag("tag")  //给以上设置的参数打标记
     *                 .getTag("tag")  //根据tag获得沉浸式参数
     *                 .init();  //必须调用方可应用以上所配置的参数
     */
    void configImmersionBar(ImmersionBar immersionBar);

    /**
     * 这个 Activity 是否会使用 Fragment,框架会根据这个属性判断是否注册 {@link FragmentManager.FragmentLifecycleCallbacks}
     * 如果返回{@code false},那意味着这个 Activity 不需要绑定 Fragment,那你再在这个 Activity 中绑定继承于 {@link BaseFragment} 的 Fragment 将不起任何作用
     *
     * @return
     * @see ActivityLifecycle# (Fragment 的注册过程)
     */
    boolean useFragment();

    void initIntent();

    void initTopBar();

    void initLanguage();

    void initLayoutParams();

    void initLayoutVisible();


}
