package com.gingersoft.supply_chain.mvp.presenter;

import android.app.Application;
import android.util.Log;

import com.gingersoft.gsa.cloud.common.utils.CollectionUtils;
import com.gingersoft.gsa.cloud.common.utils.other.TextUtil;
import com.gingersoft.gsa.cloud.ui.bean.view.CategoryBean;
import com.gingersoft.supply_chain.mvp.bean.BuyIngredientsBean;
import com.gingersoft.supply_chain.mvp.bean.CategoryFoodTreeBean;
import com.gingersoft.supply_chain.mvp.bean.FoodByCategoryResultBean;
import com.gingersoft.supply_chain.mvp.bean.OrderCategoryBean;
import com.gingersoft.supply_chain.mvp.bean.PurchaseFoodBean;
import com.gingersoft.supply_chain.mvp.content.Constant;
import com.jess.arms.integration.AppManager;
import com.jess.arms.di.scope.FragmentScope;
import com.jess.arms.mvp.BasePresenter;
import com.jess.arms.http.imageloader.ImageLoader;

import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.annotations.NonNull;
import io.reactivex.schedulers.Schedulers;
import lombok.val;
import me.jessyan.rxerrorhandler.core.RxErrorHandler;
import me.jessyan.rxerrorhandler.handler.ErrorHandleSubscriber;

import javax.inject.Inject;

import com.gingersoft.supply_chain.mvp.contract.BuyIngredientsContract;
import com.jess.arms.utils.RxLifecycleUtils;
import com.qmuiteam.qmui.widget.section.QMUISection;
import com.xuexiang.rxutil2.rxjava.RxJavaUtils;
import com.xuexiang.rxutil2.rxjava.task.RxIOTask;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Description: 選購食材
 */
@FragmentScope
public class BuyIngredientsPresenter extends BasePresenter<BuyIngredientsContract.Model, BuyIngredientsContract.View> {
    @Inject
    RxErrorHandler mErrorHandler;
    @Inject
    Application mApplication;
    @Inject
    ImageLoader mImageLoader;
    @Inject
    AppManager mAppManager;

    /**
     * 緩存分類，區分分類級別
     * 用於顯示
     */
    private List<OrderCategoryBean.FoodCategoryTrees> foodCategoryTrees;
    /**
     * 存儲所有分類id和名字，包括下面的所有食材，如果沒有食材，需要調用獲取食材接口然後存放在這個集合裡面
     * 之後點擊一個分類，可以通過這個集合去拿到之前已經通過接口獲取到的食材信息
     * <p>
     * 用於快速獲取某個分類下的食材
     */
    private List<CategoryFoodTreeBean> categoryFoodTreeBeans;

    private List<BuyIngredientsBean> buyIngredientsBeans;
//    private Map<Integer, List<QMUISection<CategoryBean, PurchaseFoodBean>>> foodSectionMap = new HashMap<>();
    /**
     * 是否正在加載食品中
     */
    private boolean isLoadFood = false;

    /**
     * 用戶選購的食材列表
     */
    private List<PurchaseFoodBean> purchaseFoodBeans = new ArrayList<>();

    @Inject
    public BuyIngredientsPresenter(BuyIngredientsContract.Model model, BuyIngredientsContract.View rootView) {
        super(model, rootView);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        this.mErrorHandler = null;
        this.mAppManager = null;
        this.mImageLoader = null;
        this.mApplication = null;
    }


    /**
     * 獲取分類的結構：所有分類和分類的子分類都有
     */
    public void getCategoryTrees() {
        if (foodCategoryTrees != null) {
            mRootView.loadCategory(foodCategoryTrees);
            return;
        }
        Map<String, Object> map = new HashMap<>(2);
        Constant.addBrandId(map);
        Constant.addRestaurantId(map);
        mModel.getCategoryTrees(map)
                .subscribeOn(Schedulers.io())
                .doOnSubscribe(disposable -> mRootView.showLoading(Constant.GET_INFO_LOADING))
                .subscribeOn(AndroidSchedulers.mainThread())
                .observeOn(AndroidSchedulers.mainThread())
                .doAfterTerminate(() -> mRootView.hideLoading())
                .compose(RxLifecycleUtils.bindToLifecycle(mRootView))
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new ErrorHandleSubscriber<OrderCategoryBean>(mErrorHandler) {
                    @Override
                    public void onNext(OrderCategoryBean orderCategoryBean) {
                        if (orderCategoryBean.isSuccess()) {
                            if (categoryFoodTreeBeans == null) {
                                categoryFoodTreeBeans = new ArrayList<>();
                            }
                            List<OrderCategoryBean.FoodCategoryTrees> data = orderCategoryBean.getData();
                            if (data != null) {
                                //將分類食品的list容量設置為一級分類的數量，這樣就不用擔心之後加載不同position的分類數據時，計算位置了
                                categoryFoods = new ArrayList<>(data.size());
                                for (OrderCategoryBean.FoodCategoryTrees ignored : data) {
                                    categoryFoods.add(null);
                                }
                            }
                            conversionCategoryTree(data);
                            foodCategoryTrees = new ArrayList<>();
                            foodCategoryTrees.addAll(data);
                            mRootView.loadCategory(data);
                        } else if (TextUtil.isNotEmptyOrNullOrUndefined(orderCategoryBean.getErrMsg())) {
                            mRootView.showMessage(orderCategoryBean.getErrMsg());
                        } else {
                            mRootView.showMessage(Constant.GET_INFO_ERROR);
                        }
                    }

                    /**
                     * 將分類轉換為categoryFoodTreeBeans，方便之後存取分類和分類下的食品
                     */
                    private void conversionCategoryTree(List<OrderCategoryBean.FoodCategoryTrees> data) {
                        if (data == null) {
                            return;
                        }
                        RxJavaUtils.doInIOThread(new RxIOTask<Object>(0) {
                            @Override
                            public Void doInIOThread(Object o) {
                                for (OrderCategoryBean.FoodCategoryTrees category : data) {
                                    CategoryFoodTreeBean categoryFoodTreeBean = new CategoryFoodTreeBean();
                                    List<Integer> categoryId = new ArrayList<>();
                                    List<String> categoryName = new ArrayList<>();
                                    categoryId.add(category.getId());
                                    categoryName.add(category.getName());
                                    if (category.getFoodCategoryTrees() != null) {
                                        for (OrderCategoryBean.FoodCategoryTrees secondCategory : category.getFoodCategoryTrees()) {
                                            categoryId.add(secondCategory.getId());
                                            categoryName.add(secondCategory.getName());
                                            if (secondCategory.getFoodCategoryTrees() != null) {
                                                for (OrderCategoryBean.FoodCategoryTrees thirdCategory : secondCategory.getFoodCategoryTrees()) {
                                                    categoryId.add(thirdCategory.getId());
                                                    categoryName.add(thirdCategory.getName());
                                                }
                                            }
                                        }
                                    }
                                    categoryFoodTreeBean.setCategoryIds(categoryId);
                                    categoryFoodTreeBean.setCategoryNames(categoryName);
                                    categoryFoodTreeBeans.add(categoryFoodTreeBean);
                                }
                                return null;
                            }
                        });
                    }
                });
    }

    public void getFoodByCategoryId() {

    }


    public void getFood(int parentCategoryId) {
//        List<QMUISection<CategoryBean, PurchaseFoodBean>> foodSections = foodSectionMap.get(parentCategoryId);
//        if (foodSections == null) {
//            getFoodByCategoryId(parentCategoryId);
//        } else {
//            mRootView.loadFood(parentCategoryId, foodSections);
//        }
    }

    private List<List<BuyIngredientsBean>> categoryFoods = new ArrayList<>();
    private List<List<BuyIngredientsBean>> supplierFoods = new ArrayList<>();

    public void clearCategoryFoods() {
        int categorySize = categoryFoods.size();
        for (int i = 0; i < categorySize; i++) {
            categoryFoods.set(i, null);
        }
    }

    public boolean isHasLocationCategoryFoodInfo(int position) {
        if (position >= categoryFoods.size()) {
            return false;
        }
        return categoryFoods.get(position) != null;
    }

    /**
     * 根據分類的下標獲得他的一級分類
     */
    public int getFirstCategoryByIndex(int index) {
        int progress = 0;
        //一級分類的下標
        int categoryIndex = 0;
        for (List<BuyIngredientsBean> categoryFood : categoryFoods) {
            if (CollectionUtils.isNotNullOrEmpty(categoryFood)) {
                if (progress <= index) {
                    progress += categoryFood.size();
                    if (progress > index) {
                        return categoryIndex;
                    }
                } else {
                    return categoryIndex;
                }
            }
            categoryIndex++;
        }
        return -1;
    }

//    public void getLocationFood(){
//        if (categoryFoods.get(position) != null) {
//            Log.e("eee", "本地獲取");
//            //將所有的兩層集合轉為一層
//            List<BuyIngredientsBean> allShowFood = getAllShowFood();
//            mRootView.scrollToPosition(getIndexInAllFood(allShowFood, categoryTrees));
//            return;
//        }
//    }

    /**
     * 根據一級分類獲取下面所有包括子分類的食品
     */
    public void getFoodByCategory(int position, OrderCategoryBean.FoodCategoryTrees categoryTrees) {
        //先看看這個分類的食材有沒有緩存，沒有緩衝再加載
        Map<String, Object> map = new HashMap<>(5);
        map.put("pageSize", 1000);
        map.put("pageIndex", 0);
        Constant.addBrandId(map);
        Constant.addRestaurantId(map);
        map.put("parentId", categoryTrees.getId());
        mModel.getFoodByCategory(map)
                .subscribeOn(Schedulers.io())
                .doOnSubscribe(disposable -> {
//                    mRootView.showLoading(Constant.GET_INFO_LOADING)
                })
                .subscribeOn(AndroidSchedulers.mainThread())
                .observeOn(AndroidSchedulers.mainThread())
                .doAfterNext(dis -> mRootView.hideLoading())
                .compose(RxLifecycleUtils.bindToLifecycle(mRootView))
                .subscribe(new ErrorHandleSubscriber<FoodByCategoryResultBean>(mErrorHandler) {

                    @Override
                    public void onNext(@NonNull FoodByCategoryResultBean info) {
                        Log.e("eee", "網絡獲取：" + info.isSuccess() + "分類名：" + categoryTrees.getName());
                        if (info.isSuccess()) {
                            //保存分類下食品
                            savFoodToCategory(info);
                            //拿到所有食品
                            List<PurchaseFoodBean> foodBeans = info.getData();
                            //初始化分類和食品的顯示集合
                            List<BuyIngredientsBean> buyIngredientsBeans = new ArrayList<>();
                            conversionShowInfo(foodBeans, categoryTrees, buyIngredientsBeans, 0);
                            //將數據插入緩衝中
                            categoryFoods.set(position, buyIngredientsBeans);
                            //將所有的兩層集合轉為一層
                            List<BuyIngredientsBean> allShowFood = getAllShowFood();
                            //加載食品
                            mRootView.loadFood(getIndexInAllFood(allShowFood, categoryTrees), allShowFood);
                        } else if (TextUtil.isNotEmptyOrNullOrUndefined(info.getErrMsg())) {
                            mRootView.showMessage(info.getErrMsg());
                            mRootView.loadFail();
                        } else {
                            mRootView.loadFail();
                        }
                    }

                    @Override
                    public void onError(Throwable t) {
                        super.onError(t);
                        Log.e("eee", "網絡獲取出錯：分類名：" + categoryTrees.getName());
                    }

                    private void conversionShowInfo(List<PurchaseFoodBean> foodBeans, OrderCategoryBean.FoodCategoryTrees foodCategoryTree, List<BuyIngredientsBean> buyIngredientsBeans, int foodSize) {
                        BuyIngredientsBean buyIngredientsBean = new BuyIngredientsBean();
                        buyIngredientsBean.id = foodCategoryTree.getId();
                        buyIngredientsBean.parentId = foodCategoryTree.getParentId();
                        buyIngredientsBean.categoryName = foodCategoryTree.getName();
                        buyIngredientsBean.purchaseFoodList = new ArrayList<>();
                        if (foodBeans != null && foodSize != foodBeans.size()) {
                            boolean haveThisCategoryBean = true;
                            for (PurchaseFoodBean foodBean : foodBeans) {
                                if (foodBean.getFoodCategoryId() == foodCategoryTree.getId()) {
                                    buyIngredientsBean.purchaseFoodList.add(foodBean);
                                    haveThisCategoryBean = false;
                                    foodSize++;
                                } else if (!haveThisCategoryBean) {
                                    //因為食品是按順序排列，如果之前已經有這個分類的食品，後來沒了，之後的數據就不用遍歷了
                                    break;
                                }
                            }
                        }
                        buyIngredientsBeans.add(buyIngredientsBean);
                        List<OrderCategoryBean.FoodCategoryTrees> foodCategoryTrees = foodCategoryTree.getFoodCategoryTrees();
                        if (foodCategoryTrees != null) {
                            for (OrderCategoryBean.FoodCategoryTrees categoryTree : foodCategoryTrees) {
                                conversionShowInfo(foodBeans, categoryTree, buyIngredientsBeans, foodSize);
                            }
                        }
                    }


                    /**
                     *  保存食品信息到分類數據中
                     * @param info
                     */
                    private void savFoodToCategory(@NonNull FoodByCategoryResultBean info) {
                        RxJavaUtils.doInIOThread(new RxIOTask<FoodByCategoryResultBean>(info) {
                            @Override
                            public Void doInIOThread(FoodByCategoryResultBean info) {
                                //將本次獲取到的食品存入到緩存中
                                for (CategoryFoodTreeBean categoryFoodTreeBean : categoryFoodTreeBeans) {
//                                    if (categoryFoodTreeBean.getCategoryIds().contains(parentCategoryId)) {
//                                        categoryFoodTreeBean.setFoodInfo(info.getData());
//                                        break;
//                                    }
                                }
                                return null;
                            }
                        });
                    }
                });
    }

    private List<BuyIngredientsBean> getAllShowFood() {
        List<BuyIngredientsBean> buyIngredientsBeans = new ArrayList<>();
        for (List<BuyIngredientsBean> categoryFood : categoryFoods) {
            if (categoryFood != null) {
                buyIngredientsBeans.addAll(categoryFood);
            }
        }
        return buyIngredientsBeans;
    }

    private int getIndexInAllFood(List<BuyIngredientsBean> allShowFood, OrderCategoryBean.FoodCategoryTrees categoryTrees) {
        //從一層中找到當前分類的位置
        int index = allShowFood.indexOf(new BuyIngredientsBean(categoryTrees.getId(), categoryTrees.getName(), categoryTrees.getParentId()));
        int position = index;
        for (int i = 0; i < index; i++) {
            position += allShowFood.get(i).purchaseFoodList.size();
        }
        return position;
    }

    /**
     * 將食品列表轉為帶分類用於顯示的列表
     *
     * @return
     */
    public List<QMUISection<CategoryBean, PurchaseFoodBean>> foodToList(OrderCategoryBean.FoodCategoryTrees categoryTrees, List<PurchaseFoodBean> data) {
        if (data == null) {
            return null;
        }
        BuyIngredientsBean buyIngredientsBean = new BuyIngredientsBean();

        List<QMUISection<CategoryBean, PurchaseFoodBean>> sections = new ArrayList<>();
        CategoryBean categoryBean = new CategoryBean();
        List<PurchaseFoodBean> purchaseFoodBeans = new ArrayList<>();
        for (int i = data.size() - 1; i >= 0; i--) {
            PurchaseFoodBean purchaseFoodBean = data.get(i);
            if (categoryBean.getId() == null) {
                categoryBean.setId(purchaseFoodBean.getFoodCategoryId());
            } else {
                if (!categoryBean.getId().equals(purchaseFoodBean.getFoodCategoryId())) {
                    sections.add(new QMUISection<>(categoryBean, purchaseFoodBeans));
                    categoryBean = new CategoryBean();
                    categoryBean.setId(purchaseFoodBean.getFoodCategoryId());
                    purchaseFoodBeans = new ArrayList<>();
                }
            }
            purchaseFoodBeans.add(purchaseFoodBean);
            if (i == 0) {
                //最後一個食品
                sections.add(new QMUISection<>(categoryBean, purchaseFoodBeans));
            }
        }
        //由於食的數據中沒有分類名稱，需要去分類的緩存中取出對應的分類名
        for (QMUISection<CategoryBean, PurchaseFoodBean> section : sections) {
            CategoryBean header = section.getHeader();
            for (CategoryFoodTreeBean categoryFoodTreeBean : categoryFoodTreeBeans) {
                if (categoryFoodTreeBean.getCategoryIds().contains(header.getId())) {
                    header.setCategoryName(categoryFoodTreeBean.getCategoryNames().get(categoryFoodTreeBean.getCategoryIds().indexOf(header.getId())));
                }
            }
        }
        return sections;
    }


    /**
     * 篩選出這個分類下的食材
     *
     * @param categoryId 分類id
     * @param foodInfo   所有食材
     * @return 當前分類下的食材
     */
    private @NotNull List<PurchaseFoodBean> filterThisCategoryFood(Integer categoryId, List<PurchaseFoodBean> foodInfo) {
        //從所有食材中篩選出自己分類下的食材
        List<PurchaseFoodBean> foods = new ArrayList<>();
        //用於提前終止循環，如果已經拿到過這個分類下的數據，後面沒有這個分類下的數據了，就終止循環
        boolean hasFood = false;
        for (PurchaseFoodBean purchaseFoodBean : foodInfo) {
            if (categoryId.equals(purchaseFoodBean.getFoodCategoryId())) {
                hasFood = true;
                foods.add(purchaseFoodBean);
            } else if (hasFood) {
                break;
            }
        }
        return foods;
    }


    public boolean isLoadFood() {
        return isLoadFood;
    }

    public void setLoadFood(boolean loadFood) {
        isLoadFood = loadFood;
    }
}
