/*
 * Tencent is pleased to support the open source community by making QMUI_Android available.
 *
 * Copyright (C) 2017-2018 THL A29 Limited, a Tencent company. All rights reserved.
 *
 * Licensed under the MIT License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 *
 * http://opensource.org/licenses/MIT
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gingersoft.gsa.cloud.common.ui.recylcler.decorator;

import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Rect;
import android.view.View;

import androidx.annotation.ColorInt;
import androidx.recyclerview.widget.GridLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

/**
 * @author cginechen
 * @date 2016-10-21
 */

public class GridDividerItemDecoration extends RecyclerView.ItemDecoration {
    private Paint mPaint;
    private int mDividerWidth;

    public GridDividerItemDecoration(int height, @ColorInt int color) {
        mDividerWidth = height;
        mPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mPaint.setColor(color);
        mPaint.setStyle(Paint.Style.FILL);
    }

    @Override
    public void getItemOffsets(Rect outRect, View view, RecyclerView parent, RecyclerView.State state) {
        super.getItemOffsets(outRect, view, parent, state);


        int itemPosition = ((RecyclerView.LayoutParams) view.getLayoutParams()).getViewLayoutPosition();
        int spanCount = getSpanCount(parent);

        boolean isfirsColumn = isfirsColumn(itemPosition, spanCount);
        boolean isLastColumn = isLastColumn(itemPosition, spanCount);

        boolean isfirstRow = isfirstRow(itemPosition, spanCount);

        int top;
        int left;
        int right;
        int bottom;

        int eachWidth = (spanCount - 1) * mDividerWidth / spanCount;

        int dl = mDividerWidth - eachWidth;

        left = itemPosition % spanCount * dl;

        right = eachWidth - left;


        bottom = mDividerWidth;

        if (isfirstRow) {
            top = mDividerWidth;
        } else {
            top = 0;
        }

        if (isfirsColumn) {
            left = mDividerWidth;
        }
        if (isLastColumn) {
            right = mDividerWidth;
        }


        outRect.set(left, top, right, bottom);
    }

    @Override
    public void onDraw(Canvas c, RecyclerView parent, RecyclerView.State state) {
        super.onDraw(c, parent, state);
        draw(c, parent);
    }

    private void draw(Canvas canvas, RecyclerView parent) {
        int childSize = parent.getChildCount();
        for (int i = 0; i < childSize; i++) {
            View child = parent.getChildAt(i);
            RecyclerView.LayoutParams layoutParams = (RecyclerView.LayoutParams) child.getLayoutParams();

            /**
             * 画水平分隔线
             */
            int left = child.getLeft();
            int right = child.getRight();
            int top = child.getBottom() + layoutParams.bottomMargin;
            int bottom = top + mDividerWidth;
            canvas.drawRect(left, top, right, bottom, mPaint);


            /**
             * 画垂直分割线
             */
            top = child.getTop();
            bottom = child.getBottom() + mDividerWidth;
            left = child.getRight() + layoutParams.rightMargin;
            right = left + mDividerWidth;
            canvas.drawRect(left, top, right, bottom, mPaint);

            int spanCount = getSpanCount(parent);

            /**
             * 如果是第一行
             */
            if (isfirstRow(i, spanCount)) {
                canvas.drawRect(0, 0, right, mDividerWidth, mPaint);
            }

            /**
             * 如果是第一列
             */
            if (isfirsColumn(i, spanCount)) {
                canvas.drawRect(0, 0, mDividerWidth, bottom, mPaint);
            }

        }
    }

    /**
     * 判断是不是第一行
     *
     * @param pos
     * @param spanCount
     * @return
     */
    private boolean isfirstRow(int pos, int spanCount) {
        if ((pos / spanCount + 1) == 1) {
            return true;
        } else {
            return false;
        }
    }


    /**
     * 判断是不是第一列
     *
     * @param pos
     * @param spanCount
     * @return
     */
    private boolean isfirsColumn(int pos, int spanCount) {
        if (pos % spanCount == 0) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 判断是不是最后一列
     *
     * @param pos
     * @param spanCount
     * @return
     */
    private boolean isLastColumn(int pos, int spanCount) {
        if ((pos - spanCount + 1) % spanCount == 0) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 列数
     *
     * @param parent
     * @return
     */
    private int getSpanCount(RecyclerView parent) {
        RecyclerView.LayoutManager layoutManager = parent.getLayoutManager();
        return ((GridLayoutManager) layoutManager).getSpanCount();
    }
}
